/**************************************************************************
MODULE:    RA_App
CONTAINS:  Main module for CANopenIA Remote Access demo implementation
COPYRIGHT: Embedded Systems Academy, Inc. 2008-2021.
DISCLAIM:  Read and understand our disclaimer before using this code!
           www.esacademy.com/disclaim.htm
           This software was written in accordance to the guidelines at
           www.esacademy.com/software/softwarestyleguide.pdf
LICENSE:   Free to use with licensed CANopenIA chips, modules or devices.
           This version optimized for COIA-M0.
VERSION:   2.00, EmSA 13-SEP-21
***************************************************************************/ 

#ifdef WIN32
#include <windows.h>
#include <conio.h>
#endif

#include <stdio.h>
#include <time.h>
#include <signal.h>
#include "SerialProtocol.h"


/**************************************************************************
MODULE CONFIGURATION
***************************************************************************/

// Select bitrate to switch to, 1 to keep at 115200, 3 for 460800, 4 for 614400
#define COIA_BITRATE 1

// Define to 1 to enable simulated data production
#define PRODUCE_DATA 1

// Define to 1 to enable display of new data on the network
#define SHOW_NEW_DATA 1


/**************************************************************************
MODULE DEFINITIONS
***************************************************************************/

// Own NMT states
#define NMTSTATE_BOOT 0
#define NMTSTATE_STOP 4
#define NMTSTATE_OP 5
#define NMTSTATE_PREOP 127


/**************************************************************************
MACROS FOR PLATFORM-ENDIANNESS-INDEPENDENT LITTLE-ENDIAN MEMORY ACCESS
**************************************************************************/
#define GEN_RD32(ptr)  ( ((uint32_t)(*((uint8_t *)ptr+3))        << 24) | \
                         ((uint32_t)(*((uint8_t *)ptr+2) & 0xFF) << 16) | \
                         ((uint32_t)(*((uint8_t *)ptr+1) & 0xFF) <<  8) | \
                          (uint32_t)(*((uint8_t *)ptr  ) & 0xFF)        )

#define GEN_RD16(ptr)  ( ((uint16_t)(*((uint8_t *)ptr+1))        << 8) |  \
                          (uint16_t)(*((uint8_t *)ptr  ) & 0xFF)       )


/**************************************************************************
MODULE VARIABLES
***************************************************************************/ 

static SerialProtocol *COIADevice = new SerialProtocol();

static uint8_t MyNodeID = 0;                 // Our own node ID
static uint8_t MyNMTState = NMTSTATE_BOOT;   // Our own state
static uint8_t TerminationRequested = FALSE; // App termination flag

// Data to read or write, here 2 of each digital, 4 analog in/out
static uint8_t  OD600001_DigIn;
static uint8_t  OD600002_DigIn;
static uint8_t  OD620001_DigOut;
static uint8_t  OD620002_DigOut;
static uint16_t OD640101_AnaIn;
static uint16_t OD640102_AnaIn;
static uint16_t OD640103_AnaIn;
static uint16_t OD640104_AnaIn;
static uint16_t OD641101_AnaOut;
static uint16_t OD641102_AnaOut;
static uint16_t OD641103_AnaOut;
static uint16_t OD641104_AnaOut;


/**************************************************************************
DOES:    Executes a write to the Object Dictionary of the COIA-M0
RETURNS: Nothing
**************************************************************************/
static void WriteOD(
	uint16_t Idx, // OD Index
	uint8_t Sub,  // OD Subindex
	uint8_t Len,  // Length of data
	uint8_t *Dat  // Pointer to data
	)
{
	uint32_t result;

	result = COIADevice->WriteLocalOD(Idx,Sub,Len,Dat);
	if (result == ERROR_NOERROR)
	{
		if (Len == 1)
		{
			printf("OUT[%4.4X,%2.2X:%2.2X] ",Idx,Sub,*Dat);
		}
		else if (Len == 2)
		{
			printf("OUT[%4.4X,%2.2X:%4.4X] ",Idx,Sub,GEN_RD16(Dat));
		}
		else if (Len == 4)
		{
			printf("OUT[%4.4X,%2.2X:%8.8X] ",Idx,Sub,GEN_RD32(Dat));
		}
		else
		{
			printf("OUT[%4.4X,%2.2X,LEN:%d] ", Idx, Sub, Len);
		}
	}
	else
	{
		printf("OUT[%4.4X,%2.2X,ERR:%8.8X] ",Idx,Sub,result);
	}
	COIADevice->Process(); // Keep RA process alive
}


/*******************************************************************************
DOES:    Called when user presses Ctrl-C. Sets a flag
RETURNS: Nothing
*******************************************************************************/
static void Terminate
(
	int SignalNumber
)
{
	TerminationRequested = TRUE;
}


/**************************************************************************
DOES:    This function is called from the new data recived call-back,
         if data received indicates a change in this devices node status:
         Change of own node ID or own NMT state or own HW status
GLOBALS: Updates MyNodeID and MyNMTState
**************************************************************************/
void OwnStatusChanged (
  uint8_t SubIdx, // 1 to 3 for node id, NMT status or HW status
  uint8_t data
  )
{
  if (SubIdx == 1)
  {
    printf("\n{Own node ID changed to %d} ", data);
    MyNodeID = data;
  }
  else if (SubIdx == 2)
  {
    printf("\n{Own status changed to %d} ", data);
    MyNMTState = data;
  }
  else if  (SubIdx == 3)
  {
    printf("\n{Hardware status changed to 0x%2.2X - ", data);
  
    if (data == HWSTATUS_NONE)
    {
      printf("NONE} ");
      return;
    }

    if (data & HWSTATUS_INITALIZING)    printf("INIT ");
    if (data & HWSTATUS_CANERROR)       printf("CAN-ERROR ");
    if (data & HWSTATUS_ERRORPASSIVE)   printf("ERROR-PASSIVE ");
    if (data & HWSTATUS_RXQUEUEOVERRUN) printf("RX-OVERRUN ");
    if (data & HWSTATUS_TXQUEUEOVERRUN) printf("TX-OVERRUN ");
    if (data & HWSTATUS_TXBUSY)         printf("TX-BUSY ");
    if (data & HWSTATUS_BUSOFF)         printf("BUS-OFF ");
    printf("} ");
  }
  else
  {
    printf("\n{Unknown Subindex: %d} ", SubIdx);
  }
}


/**************************************************************************
DOES:    This function is called from the new data recived call-back,
         if data received indicates a change in the node status of any of
         the nodes connected to the network
RETURNS: Nothing
**************************************************************************/
void NodeStatusChanged (
  uint8_t NodeID, // node ID for which a change of state was detected
                  // highest bit set if it is for own node ID
  uint8_t State   // current state of that node
  )
{
  printf("\n{Node %d ", NodeID & 0x7F);
  printf("status changed to 0x%2.2X - ",State);

  switch(State)
  {
    case NODESTATUS_BOOT: printf("BOOT} "); break;
    case NODESTATUS_EMCY_OVER: printf("EMCY CLEAR} "); break;  
    case NODESTATUS_EMCY_NEW: printf("NEW EMCY} "); break;   
    default: printf("UNKNOWN} "); break;
  }
}


/**************************************************************************
DOES:    Call-back function, data indication, new data arrived in device
RETURNS: Nothing
**************************************************************************/
void NewData (
  uint8_t  NodeID,     // node id from which this data arrived
  uint16_t Index,      // Index of Object Dictionary entry written to
  uint8_t  Subindex,   // Subindex of Object Dictionary entry written to
  uint32_t DataLength, // data length of data written
  uint8_t  *Data       // pointer to the data written
  )
{
  uint16_t *p16;

  // is this our own status that changed?
  if (Index == 0x5F00)
  { // device status
    OwnStatusChanged(Subindex,*Data);
  }
  // is this a node status change from nodes on network?
  else if (Index == 0x5F04)
  { // node status
    NodeStatusChanged(Subindex,*Data);
  }
  else if ((Index == 0x6200) && (DataLength == 1))
  { // Digital Output received
	  if (Subindex == 1)
	  {
		  OD620001_DigOut = *Data;
	  }
	  else if (Subindex == 2)
	  {
		  OD620002_DigOut = *Data;
	  }
  }
  else if ((Index == 0x6411) && (DataLength == 2))
  { // Analog Output received
	  p16 = (uint16_t *) Data; // convert data pointer to 16bit pointer
	  if (Subindex == 1)
	  {
		  OD641101_AnaOut = *p16;
	  }
	  if (Subindex == 2)
	  {
		  OD641102_AnaOut = *p16;
	  }
	  if (Subindex == 3)
	  {
		  OD641103_AnaOut = *p16;
	  }
	  if (Subindex == 4)
	  {
		  OD641104_AnaOut = *p16;
	  }
  }
#if SHOW_NEW_DATA == 1
  else
  { // display raw data received
    printf("{%4.4X,%2.2X;", Index, Subindex);
    while(DataLength > 0)
    {
      printf(" %2.2X", *Data);
      Data++;
      DataLength--;
    }
    printf("} ");
  }
#endif // SHOW_NEW_DATA
}


/**************************************************************************
DOES:    Work on CANopen communicated process data
RETURNS: Nothing
**************************************************************************/
void ProcessData(void)
{
#if PRODUCE_DATA == 1
	if (MyNMTState == NMTSTATE_OP)
	{ // we are operational

	    // DEMO: Echo first digital data byte
		OD600001_DigIn = OD620001_DigOut;
		WriteOD(0x6000, 1, 1, &(OD600001_DigIn));

		// DEMO: Echo first analog value
		OD640101_AnaIn = OD641101_AnaOut;
		WriteOD(0x6401, 1, 2, (uint8_t *) &(OD640101_AnaIn));

		// Demo: Analog in 4 shows a counter
		OD640104_AnaIn++;
		WriteOD(0x6401, 4, 2, (uint8_t *) &(OD640104_AnaIn));
	}
#endif
}


/**************************************************************************
DOES:    Identifies the COIA-M0 device connected
RETURNS: Number of errors occured during read of ID
**************************************************************************/
uint8_t IdentifiyCOIA(void)
{
	uint8_t ret_val = 0;

	unsigned long result;
	static unsigned long Length;
	static uint8_t DataBuf[20];
	uint32_t *p32 = (uint32_t *) &(DataBuf[0]);

	// Get Basic information from device
	printf("\n   Read COIA version: ");
	result = COIADevice->ReadLocalOD(0x5F00, 0x04, &Length, &(DataBuf[0]));
	if (result == ERROR_NOERROR)
	{
		printf("0x%8.8X", p32[0]);
	}
	else
	{
		printf("Error code = 0x%8.8lX\n", result);
		printf("\nFailed to get response\n");
		ret_val++;
	}
	COIADevice->Process(); // keep checking for events from CANopenIA

	printf("\n Read Chip Serial Nr: ");
	result = COIADevice->ReadLocalOD(0x5F00, 0x05, &Length, &(DataBuf[0]));
	if (result == ERROR_NOERROR)
	{
		printf("0x%8.8X-0x%8.8X-0x%8.8X-0x%8.8X", p32[0], p32[1], p32[2], p32[3]);
	}
	else
	{
		printf("Error code = 0x%8.8lX\n", result);
		ret_val++;
	}
	COIADevice->Process(); // keep checking for events from CANopenIA

	printf("\n        Read Node ID: ");
	result = COIADevice->ReadLocalOD(0x5F00, 0x01, &Length, &(DataBuf[0]));
	if (result == ERROR_NOERROR)
	{
		printf("0x%2.2X", DataBuf[0]);
	}
	else
	{
		printf("Error code = 0x%8.8lX\n", result);
		ret_val++;
	}
	COIADevice->Process(); // keep checking for events from CANopenIA

	printf("\n           NMT state: ");
	result = COIADevice->ReadLocalOD(0x5F00, 0x02, &Length, &(DataBuf[0]));
	if (result == ERROR_NOERROR)
	{
		if (DataBuf[0] == 0)    printf("Booting");
		else if (DataBuf[0] == 0x7F) printf("Preoperational");
		else if (DataBuf[0] == 5)    printf("Operational");
		else if (DataBuf[0] == 4)    printf("Stopped");
		else                         printf("Unknown 0x%2.2X", DataBuf[0]);
	}
	else
	{
		printf("Error code = 0x%8.8lX\n", result);
		ret_val++;
	}
	COIADevice->Process(); // keep checking for events from CANopenIA

	return ret_val;

}


/**************************************************************************
DOES:    Main function, open com port, run for 10 minutes
**************************************************************************/
int main(int argc, char* argv[])
{
  time_t EndTime;
  time_t CycleTime;
  char *ComPort;

  printf("\nCANopenIA Remote Access by www.esacademy.com at 115200bps\nV2.00 of 13-SEP-2021\n\n");

  if (argc != 2)
  {
    printf("Usage: RA_App <comportnumber>\n");
    return 1;
  }

  ComPort = argv[1];

#ifdef WIN32
  printf("Connecting to COM%s port at 115200bps... ", ComPort);
#else
  printf("Connecting to %s... ", ComPort);
#endif // !WIN32
  if (!COIADevice->Connect(argv[1], 115200))
  {
#ifdef WIN32
    printf("Failed to connect to COM%s port\n", ComPort);
#else
    printf("Failed to connect to %s\n", ComPort);
#endif // !WIN32
    delete COIADevice;
    return 1;
  }
#ifdef WIN32
  printf("Connected to COM%s port\n", ComPort);
#else
  printf("Connected to %s\n", ComPort);
#endif // !WIN32
#if PRODUCE_DATA == 1
  printf("\nData marked OUT[Index,Subindex:Data] is transmitted to COIA-M0 device.");
#endif

#if SHOW_NEW_DATA == 1 
  printf("\nData in {Index,Subindex;Data}-brackets is received from COIA-M0 device.\n");
#endif // SHOW_NEW_DATA

  // register callback functions
  COIADevice->RegisterDataCallback((DATACALLBACK *)NewData, NULL);
  
  // use delay to ease test and debug
  CycleTime = time(NULL) + 1;  
  while (time(NULL) <= CycleTime)
  { // check for data to process
	COIADevice->Process();
  }

#if (COIA_BITRATE != 1)
  // Switching to higher bitrate
#if (COIA_BITRATE == 3)
  #define COIA_BITRATE_bps 460800
  printf("\n\nSwitch bitrate to 460800 bps: ");
  DataBuf[0] = 3;
#endif
#if (COIA_BITRATE == 4)
  #define COIA_BITRATE_bps 614400
  printf("\n\nSwitch bitrate to 614400 bps: ");
  DataBuf[0] = 4;
#endif

  result = COIADevice->WriteLocalOD(0x5F03, 0x01, 1, &(DataBuf[0]));
  if (result == ERROR_NOERROR)
  {
	  printf(" activated\n");
  }
  else
  {
	  printf("Error code = 0x%8.8lX\n", result);
	  printf("\nFailed to get response\n");
	  printf("Closing port...\n");
	  // disconnect from COM port, finished with COIA device
	  //delete COIADevice;
	  //return 10;
  }

  // Now disconnect, and re-connect
  COIADevice->Disconnect();
#ifdef WIN32
  printf("Connecting to COM%s port... ", ComPort);
#else
  printf("Connecting to %s... ", ComPort);
#endif // !WIN32
  if (!COIADevice->Connect(argv[1], COIA_BITRATE_bps))
  {
#ifdef WIN32
	  printf("Failed to connect to COM%s port\n", ComPort);
#else
	  printf("Failed to connect to %s\n", ComPort);
#endif // !WIN32
	  delete COIADevice;
	  return 1;
  }
#ifdef WIN32
  printf("Connected to COM%s port\n", ComPort);
#else
  printf("Connected to %s\n", ComPort);
#endif // !WIN32

  COIADevice->Process(); 
#endif // COIA_BITRATE

  if (IdentifiyCOIA() == 0)
  {

	  // execure APP for 600 seconds
	  EndTime = time(NULL) + 600;
	  printf("\n\nWaiting for data...");
	  printf("\nRunning for 10min, or until CTRL-C: ");

	  // look for Ctrl-C - note, this will affect all instances of this class at once
	  TerminationRequested = FALSE;
	  signal(SIGINT, Terminate);

	  // every 3s trigger a data input/output cycle
	  CycleTime = time(NULL) + 3;

	  // keep going until termination has been requested
	  while (!TerminationRequested)
	  {
		  COIADevice->Process(); // work on received data

#if PRODUCE_DATA == 1
	// CycleTime reached?
		  if (time(NULL) >= CycleTime)
		  { // Execute every 3s
			  CycleTime = time(NULL) + 3;
			  ProcessData(); // work on process data
		  }
#endif

		  // EndTime reached?
		  if (time(NULL) >= EndTime) break;
	  }

	  // de-register callback functions
	  COIADevice->RegisterDataCallback(NULL, NULL);
	  COIADevice->RegisterSDORequestCallbacks(NULL);

	  COIADevice->Disconnect();
#ifdef WIN32
	  printf("\nDisconnected from COM%s...\n", ComPort);
#else
	  printf("\nDisconnected from %s...\n", ComPort);
#endif // !WIN32
  }

  // disconnect from COM port, finished with COIA device
  delete COIADevice;

  return 0;
}

/*----------------------- END OF FILE ----------------------------------*/
