/********************************************************************
MODULE:    CC_HAL
CONTAINS:  Hardware abstraction layer implementation LPC54618
COPYRIGHT: 2016-2018 Embedded Systems Academy, Inc (USA) and
           Embedded Systems Academy, GmbH (Germany)
HOME:      https://www.cancrypt.net
LICENSE:   The CANcrypt software in this distribution may be used 
           freely on NXP LPC546xx devices. When used on other devices
           or hardware, a commercial license is required, contact us
           at https://www.cancrypt.net for more information.

Unless required by applicable law or agreed to in writing, 
software distributed under the License is distributed on an 
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
either express or implied.

VERSION:   0.1, 06-AUG-2018
           $LastChangedRevision: 466 $
********************************************************************/

#include "CANcrypt_includes.h"
#include "can.h"
#include "canfifo.h"
#include "fsl_rng.h"
#include <board.h>


/**************************************************************************
GLOBAL VARIABLES
***************************************************************************/ 

// Global timer/counter variable, incremented every millisecond
UNSIGNED16 volatile gTimCnt = 0;

// handle for CANcrypt
extern Cc_HANDLE CcH;

#define TICKRATE_HZ (1000)          /* 1000 ticks per second */


/**************************************************************************
LOCAL VARIABLES
***************************************************************************/ 
// current CAN message received
can_frame_t rxmsg;

/* handle for non-blocking send/receive on CAN0 */
static can_handle_t s_handle0;
/* handle for non-blocking receive for FIFO 0 */
static can_fifo_transfer_t fifo0rxfer;


/**************************************************************************
LOCAL FUNCTIONS
***************************************************************************/ 

void CAN_IRQHandler(CAN_Type *base, can_handle_t *handle, status_t status, uint32_t result, void *userData);


/**************************************************************************
PUBLIC FUNCTIONS
***************************************************************************/ 

/********************************************************************
DOES:    Process CAN HW Rx FIFO
********************************************************************/
UNSIGNED8 CCHW_CheckRxFIFO(void)
{
UNSIGNED8 status;
UNSIGNED8 ret_val = 0xFF;
  
  status = CAN_ReadRxFifo(CAN0, 0, s_handle0.rxFifoFrameBuf[0]);
  if (status == kStatus_Success)
  {
      CAN_IRQHandler(CAN0, &s_handle0, kStatus_CAN_RxFifoIdle, 0, s_handle0.userData);
      ret_val = 1;
  }
  else if (status == kStatus_CAN_RxOverflow)
  {
      CAN_IRQHandler(CAN0, &s_handle0, kStatus_CAN_RxFifoIdle, 0, s_handle0.userData);
      ret_val = 2;
  }
  
  return ret_val;
}


/********************************************************************
DOES:    Generates a random value.
NOTE:    Must be suitable for security use, shall not produce the 
         same sequence of numbers on every reset or power up!!
RETURNS: Random value
********************************************************************/
UNSIGNED32 CCHW_Rand(void)
{ 
  return RNG_GetRandomData();
}


/**************************************************************************
DOES:    This function implements a CAN receive queue. With each
         function call a message is pulled from the queue.
RETURNS: 1 Message was pulled from receive queue
         0 Queue empty, no message received
**************************************************************************/
UNSIGNED8 CCHW_PullMessage (
  CAN_MSG MEM_FAR *pReceiveBuf
  )
{
CAN_MSG MEM_BUF *pSrc;

  // Check if message is in Rx FIFO  
  pSrc = CANRXFIFO_GetOutPtr();
  if (pSrc != 0)
  {
    // copy message
    memcpy(pReceiveBuf,pSrc,sizeof(CAN_MSG));
    // copying complete, update FIFO
    CANRXFIFO_OutDone();
    return TRUE; // msg received
  }
  return FALSE; // no msg rcvd 
}


/*****************************************************************************
DOES:    Enable/Disable of CAN receive interrupt
RETURNS: Nothing
*****************************************************************************/
void CCHW_EnableCANrx (void)
{
  //CAN_EnableInterrupts(CAN0, kCAN_RxFIFO0InterruptEnable);
}

void CCHW_DisableCANrx (void)
{
  //CAN_DisableInterrupts(CAN0, kCAN_RxFIFO0InterruptEnable);
}


/**************************************************************************
DOES: CANcrypt high priotity transmit, use own tranmit buffer
***************************************************************************/ 
UNSIGNED8 CCHW_TransmitNow (
  CAN_MSG *pMsg
  )
{
  UNSIGNED8 ret_val = FALSE;
  
  return ret_val;
}


/**************************************************************************
DOES: If there is something in the transmit queue, and if the transmit
      buffer is available, copy next message from queue to transmit buffer

***************************************************************************/ 
void CCHW_CheckTxQueue (
  UNSIGNED8 max_msg_to_process
  )
{
CAN_MSG MEM_BUF *pMsg = 0;

  do
  {
    // Get next message from FIFO
    pMsg = CANTXFIFO_GetOutPtr();

    if (pMsg != 0)
    { // msg in FIFO  

      /* Due to errors on some HW setups, enforce no bit rate switch, if needed */
      // pMsg->bitratemode = kCAN_BitrateModeTypeNoSwitch;
      
      /* send the message using the transmit FIFO */
      if (CAN_WriteTxFIFO(CAN0, pMsg) == kStatus_Success)
      { // success
         if (max_msg_to_process > 0)
         {
           max_msg_to_process--;
         }
      }
      else
      { // fail
         pMsg = 0;
      }

      // Update Out pointer
      CANTXFIFO_OutDone();
    }
  } while ((pMsg != 0) && (max_msg_to_process > 0));
}


/********************************************************************
DOES:    This function implements a CAN transmit FIFO. With each
         function call a message is added to the FIFO.
RETURNS: 1 Message was added to the transmit FIFO
         0 If FIFO is full, message was not added
********************************************************************/
UNSIGNED8 CCHW_PushMessage_Internal (
  CAN_MSG *pTransmitBuf // Data structure with message to be send
  )
{
CAN_MSG *pDst; // Destination pointer

  CCHW_DisableCANrx();
  // Get next message space available in FIFO
  pDst = CANTXFIFO_GetInPtr();
  if (pDst != 0)
  {
    // Copy message to transmit queue
    memcpy(pDst,pTransmitBuf,sizeof(CAN_MSG));
    // Copy completed
    CANTXFIFO_InDone();
    CCHW_EnableCANrx();
    return TRUE;
  }
  // Overrun occured
  // ADD OWN CODE TO HANDLE OVERRUN ERROR !!!
  CCHW_EnableCANrx();
  return FALSE;
}


/**************************************************************************
DOES:    Adding a CAN message to the transmit queue
RETURNS: TRUE or FALSE if queue overrun
***************************************************************************/ 
UNSIGNED8 CCHW_PushMessage (
  CAN_MSG *pTransmitBuf // Data structure with message to be send
  )
{
UNSIGNED8 ret_val = FALSE;
  
#ifdef Cc_USE_SECURE_MSG
UNSIGNED8 sec;
CAN_MSG can_buf;

  CCHW_DisableCANrx();
  // check secure transmit for this message
  sec = Cc_Process_secMsg_Tx(&CcH,pTransmitBuf,&can_buf);
  if (sec == 1)
  { // secure message
    ret_val = CCHW_PushMessage_Internal(&can_buf);
  }
  else if (sec == 2)
  { // Message known, but we are not secure
    // do not add message to FIFO
  }
  else
  { // Message not known to CANcrypt, transmit as is
    ret_val = CCHW_PushMessage_Internal(pTransmitBuf);
  }
  CCHW_EnableCANrx();
#else
  ret_val = CCHW_PushMessage_Internal(pTransmitBuf);
#endif
  
  return ret_val;
}


/**************************************************************************
DOES:    CAN interrupt handler
         Receives messages and stores them in a buffer
**************************************************************************/
void CAN_IRQHandler(CAN_Type *base, can_handle_t *handle, status_t status, uint32_t result, void *userData)
{
  CAN_MSG *pSrc, *pDst;
  UNSIGNED8 stat;
  
    /* this callback is from CAN0 */
    if (handle == &s_handle0)
    {
        /* fifo receive completed */
        if (status == (status_t)kStatus_CAN_RxFifoIdle)
        {
            /* result holds the fifo number */
            /* get the message from handle->rxFifoFrameBuf[result] */
            pSrc = handle->rxFifoFrameBuf[result];

            // check if this is a CANcrypt related message
            stat = Cc_Process_Rx(&CcH,pSrc);
          
            if (! stat)
            { // not processed by CANcrypt
              // initialize destination pointer into FIFO
              pDst = CANRXFIFO_GetInPtr();
              
              if (pDst != 0)
              { // FIFO available

                // get contents
                memcpy(pDst,pSrc,sizeof(CAN_MSG));
                // copying is all done
                CANRXFIFO_InDone();
              }
              else
              { // overrun, message lost
              }
            }

            /* receive next message using this callback */
           CAN_TransferReceiveFifoNonBlocking(CAN0, &s_handle0, &fifo0rxfer);
        }
    }
}


/**************************************************************************
DOES:    Timer interrupt handler (1ms)
**************************************************************************/
void SysTick_Handler (void) 
{
#ifdef Cc_USE_DIGOUT

UNSIGNED32 flags;
  
  flags = CAN_GetStatusFlags(CAN0);
  if ((flags & 0x00000020ul) != 0)
  { // warn, passive or off
    Cc_DIGOUT1_ON();
  }
  if ((flags & 0x00000040ul) != 0)
  { // warn, passive or off
    Cc_DIGOUT2_ON();
  }
  if ((flags & 0x00000080ul) != 0)
  { // warn, passive or off
    Cc_DIGOUT3_ON();
  }
#endif
  
  //CCHW_CheckTxQueue(2); // handle transmission, here max 2 per call
  // now handled in main
  gTimCnt++; // increment global timer counter

}


/**************************************************************************
DOES:    This function implements the initialization of the CAN interface.
RETURNS: 1 if init is completed 
         0 if init failed, bit INIT of CCHW_GetStatus stays 0
**************************************************************************/
UNSIGNED8 CCHW_Init (
  UNSIGNED16 baudrate
  )
{
can_config_t config;
can_rx_fifo_config_t fifoconfig;

#if (TXFIFOSIZE > 0)
  // Init Tx FIFO
  CANTXFIFO_Flush();
#endif  

#if (RXFIFOSIZE > 0)
  // Init RxFIFO
  CANRXFIFO_Flush();
#endif  

#if (MGRFIFOSIZE > 0)
  // Init MGRFIFO
  CANMGRFIFO_Flush();
#endif  

    /* configure for 2Mbps data, 500Mbps nominal, CAN-FD */
    CAN_GetDefaultConfig(&config);
    config.baseAddress = 0x20010000;
    config.nominalBaudRate = 500000;
    config.dataBaudRate = 2000000;
    config.timestampClock_Hz = 100000;
    CAN_Init(CAN0, &config, SystemCoreClock);

    /* set the size of FIFO0 for CAN0 and enable */
    fifoconfig.idFilterNum = 16;
    CAN_SetRxFifoConfig(CAN0, 0, &fifoconfig, true);

    /* receive all messages into FIFO0 for CAN0 */
    CAN_SetRxGlobalMask(CAN0, kCAN_GlobalFilter_Standard_FIFO0 | kCAN_GlobalFilter_Extended_FIFO0);

    /* create a handle for CAN0 for non-blocking send/receive */
    //CAN_TransferCreateHandle(CAN0, &s_handle0, CAN_IRQHandler, NULL);
    CAN_TransferCreateHandle(CAN0, &s_handle0, NULL, NULL);

    /* enable CAN 0 */
    CAN_Enable(CAN0, true);

    /* messages received into RX FIFO 0 are non-blocking
       and when received the callback function will be called from the CAN
       receive interrupt */
    fifo0rxfer.frame = &rxmsg;
    fifo0rxfer.fifoIdx = 0;
    CAN_TransferReceiveFifoNonBlocking(CAN0, &s_handle0, &fifo0rxfer);


    /* Enable SysTick Timer */
    SysTick_Config(SystemCoreClock / TICKRATE_HZ);

  return 1;
}

/**************************************************************************
DOES:    This function reads a 1 millisecond timer tick. The timer tick
         must be a UNSIGNED16 and must be incremented once per millisecond.
RETURNS: 1 millisecond timer tick
**************************************************************************/
UNSIGNED16 CCHW_GetTime (void)
{
  return gTimCnt;
}

/**************************************************************************
DOES:    This function compares a UNSIGNED16 timestamp to the internal 
         timer tick and returns 1 if the timestamp expired/passed.
RETURNS: 1 if timestamp expired/passed
         0 if timestamp is not yet reached
NOTES:   The maximum timer runtime measurable is 0x8000 (about 32 seconds).
         For the usage in MicroCANopen that is sufficient. 
**************************************************************************/
UNSIGNED8 CCHW_IsTimeExpired (
  UNSIGNED16 timestamp
  )
{
UNSIGNED16 time_now;

  time_now = gTimCnt;
  if (time_now >= timestamp)
  {
    if ((time_now - timestamp) < 0x8000)
      return 1;
    else
      return 0;
  }
  else
  {
    if ((timestamp - time_now) >= 0x8000)
      return 1;
    else
      return 0;
  }
}

#ifdef DEBUG
void DebugOutCAN(
  UNSIGNED16 id, UNSIGNED8 len,
  UNSIGNED8 d0, UNSIGNED8 d1, UNSIGNED8 d2, UNSIGNED8 d3,
  UNSIGNED8 d4, UNSIGNED8 d5, UNSIGNED8 d6, UNSIGNED8 d7
  )
{
CAN_MSG dbg;

  dbg.id = Cc_CANID_DEBUG + id - 1;
  dbg.length = len;
  dbg.dataByte[0] = d0;
  dbg.dataByte[1] = d1;
  dbg.dataByte[2] = d2;
  dbg.dataByte[3] = d3;
  dbg.dataByte[4] = d4;
  dbg.dataByte[5] = d5;
  dbg.dataByte[6] = d6;
  dbg.dataByte[7] = d7;
  dbg.format = kCAN_FrameFormatStandard;
  dbg.type = kCAN_FrameTypeData;
  dbg.proto = kCAN_ProtoTypeFD;
  dbg.bitratemode = kCAN_BitrateModeTypeSwitch;

  CCHW_PushMessage_Internal(&dbg);
}
#endif


#ifdef CANcrypt_DEBUG_DETAIL
void DebugOutKey(
  UNSIGNED16 src, 
  UNSIGNED32 *pad,
  Cc_HANDLE *cch
  )
{
CAN_MSG dbg;

  dbg.id = 0x1F000000ul + (src << 8) + Cc_DEVICE_ID;
  dbg.length = 20;
  if (cch != NULL)
  {
    dbg.dataByte[0] = cch->key_up_cnt;
    dbg.dataByte[1] = cch->grp_key_id_mm;
    dbg.dataByte[2] = cch->grp_hb_seq;
    dbg.dataByte[3] = cch->grp_hb_seq >> 8;
    memcpy(&(dbg.dataByte[4]),cch->key_dyn,8);
  }
  else
  {
    memset(dbg.dataByte,0,12);
  }
  if (pad != NULL)
  {
    memcpy(&(dbg.dataByte[12]),pad,8);
  }
  else
  {
    memset(&(dbg.dataByte[12]),0,8);
  }
  dbg.format = kCAN_FrameFormatExtend;
  dbg.type = kCAN_FrameTypeData;
  dbg.proto = kCAN_ProtoTypeFD;
  dbg.bitratemode = kCAN_BitrateModeTypeSwitch;

  CCHW_PushMessage_Internal(&dbg);
}
#endif


/*----------------------- END OF FILE ----------------------------------*/
