/**************************************************************************
MODULE:    Cc_SecMessage.c, CANcrypt Secure Messaging Demo Module
CONTAINS:  Secure message handling
COPYRIGHT: 2016-2018 Embedded Systems Academy, Inc (USA) and
           Embedded Systems Academy, GmbH (Germany)
HOME:      https://www.cancrypt.net
LICENSE:   The CANcrypt software in this distribution may be used 
           freely on NXP LPC546xx devices. When used on other devices
           or hardware, a commercial license is required, contact us
           at https://www.cancrypt.net for more information.

Unless required by applicable law or agreed to in writing, 
software distributed under the License is distributed on an 
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
either express or implied.

VERSION:   0.1, 06-AUG-2018
           $LastChangedRevision: 466 $
********************************************************************/

#include <string.h>

#ifdef __cplusplus
extern "C" {
#endif

#include "CANcrypt_includes.h"

#ifdef Cc_USE_SECURE_MSG

#ifdef CANcrypt_DEBUG_DETAIL
 #ifndef _MSC_VER
 #warning "DEBUG ACTIVE: sharing pads and keys"
 #endif
#endif

/**************************************************************************
LOCAL DEFINITIONS
***************************************************************************/ 


/**************************************************************************
MODULE VARIABLE
***************************************************************************/ 


/**************************************************************************
LOCAL FUNCTIONS
***************************************************************************/ 

UNSIGNED8 mCC_IsMsgInList(
  COBID_TYPE canid,
  Cc_SEC_MSG_TABLE_ENTRY *pLst
  )
{
  UNSIGNED8 lp = 0;
  UNSIGNED8 ret_val = 0xFF;
  
  while(lp < 0xFF)
  {
    if ((canid & pLst[lp].CAN_ID_Mask) == pLst[lp].CAN_ID_Match)
    { // found it
      ret_val = lp;
      lp = 0xFF;
    }
    else if ((pLst[lp].CAN_ID_Match & COBID_MASKID) == COBID_MASKID)
    { // end of table
      lp = 0xFF;
    }
    else
    {
      lp++;
    }
  }
  
  return ret_val;
}


/**************************************************************************
PUBLIC FUNCTIONS
***************************************************************************/ 

/**************************************************************************
DOES:    Installs the secure message handlers by passing the appropriate
         secure message tables for transmit and receive.
RETURNS: TRUE, if tables were installed
         FALSE, if not
**************************************************************************/
UNSIGNED8 Cc_Load_Sec_Msg_Table(
  Cc_HANDLE *pCc,       // pointer to CANcrypt handle record
  Cc_SEC_MSG_TABLE_ENTRY *pMsgTblRx, // secure messages to receive
  UNSIGNED16           *pMsgTrkRCnt, // counter list for above
  Cc_SEC_MSG_TABLE_ENTRY *pMsgTblTx, // secure messages to receive
  UNSIGNED16           *pMsgTrkTcnt  // counter list for above
)
{
UNSIGNED8 ret_val = FALSE;
  
  if (pCc != NULL)
  {
    pCc->pMsgTblRx = pMsgTblRx;
    pCc->pMsgCntRx = pMsgTrkRCnt;
    pCc->pMsgTblTx = pMsgTblTx;
    pCc->pMsgCntTx = pMsgTrkTcnt;
    ret_val = TRUE;
  }

  return ret_val;
}


/**************************************************************************
DOES:    Initialize the transmit and receive counters with values
         from the dynamic key so that they do not start at zero
RETURNS: nothing
**************************************************************************/
void Cc_Init_Sec_Msg_Table_Counter(
  Cc_HANDLE *pCc       // Pointer to handle with CANcrypt data
)
{
  UNSIGNED8 lp;
  UNSIGNED8 *pB;
  
  pB = (UNSIGNED8 *) pCc->key_dyn;
  if (pCc->pMsgTblRx != NULL)
  {
    lp = 0;
    while (pCc->pMsgTblRx[lp].EncryptFirst != 0xFF)
    {
      pCc->pMsgCntRx[lp] =  pB[ pCc->pMsgTblRx[lp].Producer   &(Cc_KEY_LEN8-1)];
      pCc->pMsgCntRx[lp] += pB[(pCc->pMsgTblRx[lp].Producer+1)&(Cc_KEY_LEN8-1)];
      lp++;
    }
  }
  if (pCc->pMsgTblTx != NULL)
  {
    lp = 0;
    while (pCc->pMsgTblTx[lp].EncryptFirst != 0xFF)
    {
      pCc->pMsgCntTx[lp] =  pB[ pCc->pMsgTblTx[lp].Producer   &(Cc_KEY_LEN8-1)];
      pCc->pMsgCntTx[lp] += pB[(pCc->pMsgTblTx[lp].Producer+1)&(Cc_KEY_LEN8-1)];
      lp++;
    }
  }
  //DebugOutCAN(Cc_DEVICE_ID,6,pCc->pMsgCntTx[0],pCc->pMsgCntTx[0]>>8,pCc->pMsgCntRx[0],pCc->pMsgCntRx[0]>>8,pCc->pMsgCntRx[1],pCc->pMsgCntRx[1]>>8,0,0);
}


/********************************************************************
DOES:    Handles the encryption of CAN data in block sizes of
         Cc_KEY_LEN32
RETURNS: CAN data specified by pCc->pMsgTblTx[found] gets encrypted
********************************************************************/
void Cc_Encrypt_loop(
  Cc_HANDLE *pCc,       // Pointer to handle with CANcrypt data
  CAN_MSG *pCANtx,      // Pointer to CAN message to transmit
  Cc_SEC_MSG_TABLE_ENTRY *pMsg, // pointer into pCc->pMsgxxx list
  UNSIGNED32 factor     // key generation random factor
)
{
UNSIGNED32 pad[Cc_KEY_LEN32];
UNSIGNED8 todo;
UNSIGNED8 lp;
  
//UNSIGNED8 *p8 = (UNSIGNED8 *) &(pad[0]);

  // Sanity checks on parameters, only process, if first < length
  if ( (pCANtx->length >= 8) && ((pMsg->EncryptFirst + 8) < pCANtx->length) )
  {
    // first generate new key pad
    Ccuser_MakeKey(pCc->key_dyn,pCc->pkey_perm,factor,pad);
    //DebugOutCAN(Cc_DEVICE_ID,8,pCANtx->id&0x0F,found,factor,factor>>8,p8[4],p8[5],p8[6],p8[7]);

    // work in packets the size of Cc_KEY_LEN8
    todo = pMsg->EncryptLen;
    // Sanity check on length
    if ((pMsg->EncryptFirst + todo) > (pCANtx->length - 8))
    {
      todo = pCANtx->length - pMsg->EncryptFirst - 8;
    }

    lp = 0;
    while(todo > 0)
    {
      if (todo >= Cc_KEY_LEN8)
      {
        // encrypt next packet
        Ccuser_Encrypt(pad,(UNSIGNED32 *)pCANtx->dataByte,pMsg->EncryptFirst+(lp*Cc_KEY_LEN8),Cc_KEY_LEN8);
        // Now re-generate new pad
        Ccuser_MakeKey(pCc->key_dyn,pCc->pkey_perm,factor+todo,pad);
        todo -= Cc_KEY_LEN8;
      }
      else
      {
        Ccuser_Encrypt(pad,(UNSIGNED32 *)pCANtx->dataByte,pMsg->EncryptFirst+(lp*Cc_KEY_LEN8),todo);
        todo = 0;
      }
      lp++;
    }
    
  }
}


/********************************************************************
DOES:    This is the secure CAN transmit function of CANcrypt. It
         must be called before the transmit message is copied
         to the transmit FIFO, as a preamble might need to be 
         inserted first.
RETURNS: 0: message ignored (not handled) by CANcrypt     ADD TO FIFO
         1: message is secured by CANcrypt,  ADD PREAMBLE&MSG TO FIFO
         2: message requires security, but we are not paired
                                                   DO NOT ADD TO FIFO
********************************************************************/
UNSIGNED8 Cc_Process_secMsg_Tx(
  Cc_HANDLE *pCc,       // Pointer to handle with CANcrypt data
  CAN_MSG *pCANtx,      // Pointer to CAN message to transmit
  CAN_MSG *pCANcpy      // Pointer to sec CAN message buffer
)
{
UNSIGNED8 found;
UNSIGNED8 lp;
UNSIGNED32 rnd;
Cc_SECURITY_RECORD *pSec; // CANcryprt security record
UNSIGNED8 ret_val = 0;
  
  if ( (pCc != NULL) && (pCANtx != NULL) &&
       (pCc->pMsgTblTx != NULL) && (pCc->pMsgCntTx != NULL) 
     )
  { // all pointers available
    found = mCC_IsMsgInList(pCANtx->id,pCc->pMsgTblTx);
    if (found != 0xFF)
    {
      if (pCc->pMsgTblTx[found].Producer == pCc->my_address) 
      { // entry found
        if ((pCc->my_status & Cc_GROUP_STAT_BITS) == Cc_GROUP_STAT_GROUPED)
        { // we are paired or grouped
          ret_val = 1;
          // copy message to secure tx buffer
          memcpy(pCANcpy,pCANtx,sizeof(CAN_MSG));
          // from no on work on copy
          pCANtx = pCANcpy;
          // increase message size for security record
          if (pCANtx->length <= 4)
          {
            pSec = (Cc_SECURITY_RECORD *) &(pCANtx->dataByte[4]);
            pSec->pad_bytes = 4 - pCANtx->length;
            pCANtx->length = 12;
          }
          else if (pCANtx->length <= 8)
          {
            pSec = (Cc_SECURITY_RECORD *) &(pCANtx->dataByte[8]);
            pSec->pad_bytes = 8 - pCANtx->length;
            pCANtx->length = 16;
          }
          else if (pCANtx->length <= 12)
          {
            pSec = (Cc_SECURITY_RECORD *) &(pCANtx->dataByte[12]);
            pSec->pad_bytes = 12 - pCANtx->length;
            pCANtx->length = 20;
          }
          else if (pCANtx->length <= 16)
          {
            pSec = (Cc_SECURITY_RECORD *) &(pCANtx->dataByte[16]);
            pSec->pad_bytes = 16 - pCANtx->length;
            pCANtx->length = 24;
          }
          else if (pCANtx->length <= 24)
          {
            pSec = (Cc_SECURITY_RECORD *) &(pCANtx->dataByte[24]);
            pSec->pad_bytes = 24 - pCANtx->length;
            pCANtx->length = 32;
          }
          else if (pCANtx->length <= 40)
          {
            pSec = (Cc_SECURITY_RECORD *) &(pCANtx->dataByte[40]);
            pSec->pad_bytes = 40 - pCANtx->length;
            pCANtx->length = 48;
          }
          else if (pCANtx->length <= 56)
          {
            pSec = (Cc_SECURITY_RECORD *) &(pCANtx->dataByte[56]);
            pSec->pad_bytes = 56 - pCANtx->length;
            pCANtx->length = 64;
          }
          else
          {
            ret_val = 0;
          }
          if (ret_val)
          { 
            // fill unused bytes in message with random data
            lp = pSec->pad_bytes;
            while(lp > 0)
            {
              pCANtx->dataByte[pCANtx->length-8-lp] = CCHW_Rand();
              lp--;
            }
            
            // fill in data for security record
            rnd = CCHW_Rand();
            pSec->rnd_lo = (UNSIGNED8) rnd;
            pSec->rnd_hi = (UNSIGNED8) (rnd >> 8);
            pSec->seq_lo = (UNSIGNED8) pCc->pMsgCntTx[found];
            pSec->seq_hi = (UNSIGNED8) (pCc->pMsgCntTx[found] >> 8);
            pSec->active = 1;
            pSec->key_id = (pCc->grp_key_id_mm >> 5);
            pSec->key_cnt = pCc->key_up_cnt;
            if (pCc->pMsgTblTx[found].EncryptLen > 0)
            { // encrypt data
              pSec->encrypt = 1;
            }
            else
            { // do not encrypt
              pSec->encrypt = 0;
            }
            
            // Generate signature
            if (!Ccuser_MakeSignature(pCANtx,pSec,pCc->key_dyn,&(pCc->pkey_perm[Cc_PERMKEY_LEN32-Cc_KEY_LEN32])))
            { // What to do on signature generation error ???
            }
            
            if (pCc->pMsgTblTx[found].EncryptLen > 0)
            { // encrypt data
              Cc_Encrypt_loop(pCc,pCANtx,&(pCc->pMsgTblTx[found]),mCc_Get32bit((UNSIGNED8 *)pSec));
            }
            
            // increment transmission counter for this message
            pCc->pMsgCntTx[found] += Cc_SEC_INC;
          }
        }
        else
        { // we are not paired, do not send this message
          pCc->fctcb_event(Cc_ERR_MSG_NOPAIR,pCANtx->id,0);
          ret_val = 2;
        }
      }
    }
  }

  return ret_val;
}


/********************************************************************
DOES:    Handles the decryption of CAN data in block sizes of
         Cc_KEY_LEN32
RETURNS: CAN data specified by pCc->pMsgTblRx[found] gets decrypted
********************************************************************/
void Cc_Decrypt_loop(
  Cc_HANDLE *pCc,       // Pointer to handle with CANcrypt data
  CAN_MSG *pCANrx,      // Pointer to CAN message received
  Cc_SEC_MSG_TABLE_ENTRY *pMsg, // pointer into pCc->pMsgxxx list
  UNSIGNED32 factor,    // key generation random factor
  UNSIGNED8 off         // 0 for current key, Cc_KEY_LEN32 for previous
)
{
UNSIGNED32 pad_de[Cc_KEY_LEN32];
UNSIGNED8 todo;
UNSIGNED8 lp;
  
//UNSIGNED8 *p8 = (UNSIGNED8 *) &(pad_de[0]);

  // Sanity checks on parameters, only process, if first < length
  if ( (pCANrx->length >= 8) && ((pMsg->EncryptFirst + 8) < pCANrx->length) )
  {
    // first generate new key pad
    Ccuser_MakeKey(&(pCc->key_dyn[off]),pCc->pkey_perm,factor,pad_de);
    //DebugOutCAN(Cc_DEVICE_ID,8,0x10+(pCANrx->id&0x0F),found,factor,factor>>8,p8[4],p8[5],p8[6],p8[7]);

    // work in packets the size of Cc_KEY_LEN8
    todo = pMsg->EncryptLen;
    // Sanity check on length
    if ((pMsg->EncryptFirst + todo) > (pCANrx->length - 8))
    {
      todo = pCANrx->length - pMsg->EncryptFirst - 8;
    }
    
    lp = 0;
    while(todo > 0)
    {
      if (todo >= Cc_KEY_LEN8)
      {
        // encrypt next packet
        Ccuser_Decrypt(pad_de,(UNSIGNED32 *)pCANrx->dataByte,pMsg->EncryptFirst+(lp*Cc_KEY_LEN8),Cc_KEY_LEN8);
        // Now re-generate new pad
        Ccuser_MakeKey(&(pCc->key_dyn[off]),pCc->pkey_perm,factor+todo,pad_de);
        todo -= Cc_KEY_LEN8;
      }
      else
      {
        Ccuser_Decrypt(pad_de,(UNSIGNED32 *)pCANrx->dataByte,pMsg->EncryptFirst+(lp*Cc_KEY_LEN8),todo);
        todo = 0;
      }
      lp++;
    }
  }
}


/********************************************************************
DOES:    This is the secure CAN Rx function of CANcrypt, needs to be
         called before a message received goes into receive FIFO.
RETURNS: 0: message ignored by CANcrypt,                 ADD TO FIFO
         1, message is a preamble,                DO NOT ADD TO FIFO
         2, secure message, and it is secure,            ADD TO FIFO
         3, message requires security, but we are not paired
         5, message requires security, checksum failed
                                                  DO NOT ADD TO FIFO
********************************************************************/
UNSIGNED8 Cc_Process_secMsg_Rx(
  Cc_HANDLE *pCc,       // Pointer to handle with CANcrypt data
  CAN_MSG *pCANrx       // Pointer to CAN message received
)
{
UNSIGNED8 found;
UNSIGNED8 verify;
Cc_SECURITY_RECORD *pSec; // CANcryprt security record
UNSIGNED8 ret_val = 0;

  
  if ( (pCc != NULL) && (pCANrx != NULL) &&
       (pCc->pMsgTblRx != NULL) && (pCc->pMsgCntRx != NULL) 
     )
  { // all pointers available
    found = mCC_IsMsgInList(pCANrx->id,pCc->pMsgTblRx);
    if (found != 0xFF)
    { // secure msg received
      if ( ((pCc->my_status & Cc_GROUP_STAT_BITS) == Cc_GROUP_STAT_GROUPED)
           && (pCANrx->length >= 8) 
         )
      { // secure mode active to work on it
        pSec = (Cc_SECURITY_RECORD *) &(pCANrx->dataByte[pCANrx->length-8]);
        if ( (pSec->seq_lo == (UNSIGNED8) pCc->pMsgCntRx[found]) &&
             (pSec->seq_hi == ((UNSIGNED8) (pCc->pMsgCntRx[found] >> 8)))
           )
        { // message is in sequence
        
          if (pSec->key_cnt == (pCc->key_up_cnt & 0x03))
          { // use current dynamic key
            
            if (pSec->encrypt == 1)
            { // message is encrypted, handle decryption
              Cc_Decrypt_loop(pCc,pCANrx,&(pCc->pMsgTblRx[found]),mCc_Get32bit((UNSIGNED8 *)pSec),0);
            }
            
            verify = Ccuser_VerifySignature(pCANrx,pSec,pCc->key_dyn,&(pCc->pkey_perm[Cc_PERMKEY_LEN32-Cc_KEY_LEN32]));
          }
          else
          { // use previous dynamic key
            
            if (pSec->encrypt == 1)
            { // message is encrypted, handle decryption
              Cc_Decrypt_loop(pCc,pCANrx,&(pCc->pMsgTblRx[found]),mCc_Get32bit((UNSIGNED8 *)pSec),Cc_KEY_LEN32);
            }
            
            verify = Ccuser_VerifySignature(pCANrx,pSec,&(pCc->key_dyn[Cc_KEY_LEN32]),&(pCc->pkey_perm[Cc_PERMKEY_LEN32-Cc_KEY_LEN32]));
            
#ifdef CANcrypt_DEBUG_DETAIL
        DebugOutKey(0x0A,NULL,pCc);
#endif       

          }
          
          if (verify)
          { // checksum verified
            // adapt length to original
            pCANrx->length -= (8 + pSec->pad_bytes);
            pCc->pMsgCntRx[found] += Cc_SEC_INC;
            ret_val = 2;
          }
          else
          { // message not secure
            pCc->err_counter += 31;
  #if 0
            pCANrx->id += 0x1E000000ul + (Cc_DEVICE_ID << 16);
            pCANrx->format = kCAN_FrameFormatExtend;
            CCHW_PushMessage_Internal(pCANrx);
            pCANrx->format = kCAN_FrameFormatStandard;
            pCANrx->id -= (0x1E000000ul + (Cc_DEVICE_ID << 16));
            DebugOutKey(0xEC,NULL,pCc);
  #endif      
            Cc_QueueEvent(pCc,Cc_ERR_MSG_NOSEC,pCANrx->id,0);
            ret_val = 5;
          }
        }
        else
        { // secure message out of sequence
  #if 0
            pCANrx->id += 0x1E100000ul + (Cc_DEVICE_ID << 16);
            pCANrx->format = kCAN_FrameFormatExtend;
            CCHW_PushMessage_Internal(pCANrx);
            pCANrx->format = kCAN_FrameFormatStandard;
            pCANrx->id -= (0x1E100000ul + (Cc_DEVICE_ID << 16));
            DebugOutKey(0xEC,NULL,pCc);
  #endif      
          Cc_QueueEvent(pCc,Cc_ERR_MSG_NOSEC,pCANrx->id,pCc->my_status);
          ret_val = 3;
        }
      }
      else
      { // we are not paired/grouped
        Cc_QueueEvent(pCc,Cc_ERR_MSG_NOPAIR,pCANrx->id,pCc->my_status);
        ret_val = 3;
      }
    }
  }

  return ret_val;
}


/**************************************************************************
DOES:    Main CANcrypt secure messaging householding functionality. 
         Call cyclicly.
RETURNS: TRUE, if there was something to process
         FALSE, if there was nothing to do
**************************************************************************/
UNSIGNED8 Cc_Process_secMsg_Tick(
  Cc_HANDLE *pCc        // Pointer to handle with CANcrypt data
)
{
UNSIGNED8 ret_val = FALSE;

  return ret_val;
}

#endif 

#ifdef __cplusplus
}
#endif
/*----------------------- END OF FILE ----------------------------------*/

