/********************************************************************
MODULE:    CANcrypt_api.h, API functions
CONTAINS:  Definitions for CANcrypt Application Programming Interface
AUTHORS:   Embedded Systems Academy, Inc (USA) and
           Embedded Systems Academy, GmbH (Germany)
HOME:      https://www.cancrypt.net
LICENSE:   See below, APPLIES TO THIS FILE ONLY. 
           See individual file headers for applicable license.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at
  www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or 
  implied. See the License for the specific language governing 
  permissions and limitations under the License.

VERSION:   1.03, 06-AUG-2018
           $LastChangedRevision: 465 $
********************************************************************/

#ifndef _CANCRYPT_API_H
#define _CANCRYPT_API_H

#include "CANcrypt_types.h"


/********************************************************************
DOES:    Selects the next grouping event
         If none specified, last saved session key and group is used.
RETURNS: TRUE, if key availanle, else FALSE
********************************************************************/
UNSIGNED8 Cc_SelectGroup(
  Cc_HANDLE  *pCc,      // pointer to CANcrypt handle record
  UNSIGNED8  key_major, // key ID major info (2-6)
  UNSIGNED8  key_minor, // key ID minor info (size in 32bits)
  UNSIGNED16 grp_exp ,  // Expected devices (bit 0 = unused, 
                        // bit 1 = Device1, bit 2 = Device 2, etc.)
  UNSIGNED16 grp_opt    // Optional devices
);

/********************************************************************
DOES:    Re-start of the CANcrypt system.
RETURNS: TRUE, if completed
         FALSE, if error in parameters passed
********************************************************************/
UNSIGNED8 Cc_Restart(
  Cc_HANDLE  *pCc,      // pointer to CANcrypt handle record
  UNSIGNED8   address,  // address of this device, set to zero if
                        // taken from config Ccnvol_GetGroupInfo()
  UNSIGNED32  control,  // Bit0-1: 00: No change to pairing
                        //         01: Restart pairing   
                        //         10: Stop pairing 
                        // Bit2-3: 00: No change to grouping 
                        //         01: Restart grouping
                        //         10: Stop grouping
  // Call-back functions
  Cc_CB_EVENT fct_event,      // change of status, alert
  Cc_CAN_PUSH fct_pushTxFIFO, // put CAN message into Tx FIFO
  Cc_CAN_PUSH fct_pushTxNOW,  // transmit this CAN message now
  // Device identification
  UNSIGNED32 id_1018[4] // Vendor ID, product code, revision, serial
);


/********************************************************************
DOES:    Disconnect from the CANcrypt communication parners,
         sends a request to end pairing / grouping. 
RETURNS: nothing
********************************************************************/
void Cc_TxDisconnect(
  Cc_HANDLE *pCc,       // pointer to CANcrypt handle record
  UNSIGNED8 dest_addr,  // paired device ID (1-15) or 0 for group
  UNSIGNED8 reason      // reason for disconnecting, event/aler code
);


/********************************************************************
DOES:    Generate a response message of type acknowledge or abort.
RETURNS: nothing
********************************************************************/
void Cc_TxAckAbort(
  Cc_HANDLE *pCc,       // pointer to CANcrypt handle record
  UNSIGNED8 ack,        // TRUE for Ack, FALSE for Abort
  UNSIGNED8 dest_addr,  // destination device ID (1-15) 
                        // or 0 for broadcast
  UNSIGNED8 key_id,     // key id for this acknowledge, 0 if unused
  UNSIGNED8 key_len     // key len for this acknowledge, 0 if unused
);


/********************************************************************
DOES:    Generate an alert message.
RETURNS: nothing
********************************************************************/
void Cc_TxAlert(
  Cc_HANDLE *pCc,       // pointer to CANcrypt handle record
  UNSIGNED8 dest_addr,  // destination device ID (1-15) 
                        // or 0 for broadcast
  UNSIGNED16 alert      // 16bit alert or error code
);


/********************************************************************
DOES:    Generate an identify message.
RETURNS: nothing
********************************************************************/
void Cc_TxIdentify(
  Cc_HANDLE *pCc,       // pointer to CANcrypt handle record
  UNSIGNED16 version,   // CANcrypt version
  UNSIGNED8 key_id,     // key id desired
  UNSIGNED8 key_len,    // key len desired
  UNSIGNED8 cc_method,  // method desired (Cc_SECFCT_xxx)
  UNSIGNED8 cc_timing   // timing desired (Cc_TIMING_xxx)
);


/*****************************************************************************
DOES:    Generates an extended identify message
RETURNS: nothing
*****************************************************************************/
void Cc_TxXtdIdentify(
  Cc_HANDLE *pCc,               // pointer to a CANcrypt handle record
  UNSIGNED8 device,             // device to send request to
  UNSIGNED16 index,             // index to data
  UNSIGNED8 subindex            // subindex to data
);


#ifdef Cc_USE_MONITORING
/********************************************************************
         CAN receive monitoring for unexpected messages
DOES:    Init (erase) the list of CAN IDs
RETURNS: TRUE, if list was erased
         FALSE, if list could not be erased
********************************************************************/
UNSIGNED8 Cc_Init_IDList (
  Cc_CANIDLIST_HANDLE *pIDs     // pointer to list and length
);


/********************************************************************
         CAN receive monitoring for unexpected messages
DOES:    Adds a CAN ID to the maintained list of IDs.
RETURNS: TRUE, if ID was added
         FALSE, if ID could not be added
********************************************************************/
UNSIGNED8 Cc_Addto_IDList(
  Cc_CANIDLIST_HANDLE *pIDs,    // pointer to list and length
  COBID_TYPE can_id             // CAN ID to add
);
#endif

#ifdef Cc_USE_SECURE_MSG
/********************************************************************
DOES:    Installs the secure message handlers by passing the 
         appropriate secure message tables for transmit and receive.
RETURNS: TRUE, if completed
         FALSE, if error in parameters passed
********************************************************************/
UNSIGNED8 Cc_Load_Sec_Msg_Table(
  Cc_HANDLE *pCc,       // pointer to CANcrypt handle record
  Cc_SEC_MSG_TABLE_ENTRY *pMsgTblRx, // secure messages to receive
  UNSIGNED16           *pMsgTrkRCnt, // counter list for above
  Cc_SEC_MSG_TABLE_ENTRY *pMsgTblTx, // secure messages to receive
  UNSIGNED16           *pMsgTrkTcnt  // counter list for above
);


/********************************************************************
DOES:    Initialize the transmit and receive counters for the secure
         messages, may only be called "synchronized" for all
         paired / grouped devices, e.g. directly with pairing /
         grouping confirmation.
RETURNS: nothing
********************************************************************/
void Cc_Init_Sec_Msg_Table_Counter(
  Cc_HANDLE *pCc       // pointer to CANcrypt handle record
);
#endif


/*****************************************************************************
DOES:    Transmits a secure message pair as used for secure messaging and
         generic access. CAN messages must be pre-filled with data. 
         Generates checksum, encrypts messages and transmits them.
NOTE:    Must be grouped or paired
RETURNS: TRUE, when queued
*****************************************************************************/
UNSIGNED8 Cc_TxGenAccess(
  Cc_HANDLE *pCc,               // pointer to a CANcrypt handle record
  CAN_MSG *pcan1,               // pointer to pair of CAN messages with
  CAN_MSG *pcan2,               // generic access request/response
  UNSIGNED8 *pcnt,              // message counter used
  UNSIGNED8 first,              // first data byte to encrypt in can2
  UNSIGNED8 num                 // number of data bytes to encrypt in can2
);


/********************************************************************
DOES:    Main CANcrypt householding functionality. Call cyclicly.
         Primarily monitors timeouts and satet transitions.
RETURNS: TRUE, if there was something to process
         FALSE, if there was nothing to do
********************************************************************/
UNSIGNED8 Cc_Process_Tick(
  Cc_HANDLE *pCc        // pointer to CANcrypt handle record
);


/********************************************************************
DOES:    This is the main CAN receive function of CANcrypt, must be
         called directly from CAN receive interrupt. Distributes a
         message to the other Cc_Process_xxx_Rx() functions.
RETURNS: TRUE, if this message was processed by CANcrypt
         FALSE, if this message was ignored by CANcrypt
********************************************************************/
UNSIGNED8 Cc_Process_Rx(
  Cc_HANDLE *pCc,       // pointer to CANcrypt handle record
  CAN_MSG *pCANrx       // pointer to CAN message received
);


/********************************************************************
Same as above, but for individual tasks within CANcrypt:
bit and key generation process, pairing, grouping, monitoring
********************************************************************/
UNSIGNED8 Cc_Process_Key_Tick(Cc_HANDLE *pCc);
UNSIGNED8 Cc_Process_Pair_Tick(Cc_HANDLE *pCc);
UNSIGNED8 Cc_Process_Group_Tick(Cc_HANDLE *pCc);
UNSIGNED8 Cc_Process_secMsg_Tick(Cc_HANDLE *pCc);
UNSIGNED8 Cc_Process_Monitor_Tick(Cc_HANDLE *pCc);
UNSIGNED8 Cc_Process_Key_Rx(Cc_HANDLE *pCc, CAN_MSG *pCANrx);
UNSIGNED8 Cc_Process_Pair_Rx(Cc_HANDLE *pCc, CAN_MSG *pCANrx);
UNSIGNED8 Cc_Process_Group_Rx(Cc_HANDLE *pCc, CAN_MSG *pCANrx);
UNSIGNED8 Cc_Process_Monitor_Rx(Cc_HANDLE *pCc, CAN_MSG *pCANrx);


/********************************************************************
DOES:    This is the secure CAN Rx function of CANcrypt, needs to be
         called before a message received goes into receive FIFO.
RETURNS: 0: message ignored by CANcrypt,                 ADD TO FIFO
         1, message is a preamble,                DO NOT ADD TO FIFO
         2, secure message, and it is secure,            ADD TO FIFO
         3, message requires security, but we are not paired
                                                  DO NOT ADD TO FIFO
********************************************************************/
UNSIGNED8 Cc_Process_secMsg_Rx(
  Cc_HANDLE *pCc,       // pointer to CANcrypt handle record
  CAN_MSG *pCANrx       // pointer to CAN message received
);


/********************************************************************
DOES:    This is the secure CAN transmit function of CANcrypt. It
         must be called before the transmit message is copied
         to the transmit FIFO, as a preamble might need to be 
         inserted first.
RETURNS: 0: message ignored (not handled) by CANcrypt     ADD TO FIFO
         1: message is secured by CANcrypt,  ADD PREAMBLE&MSG TO FIFO
         2: message requires security, but we are not paired
                                                   DO NOT ADD TO FIFO
********************************************************************/
UNSIGNED8 Cc_Process_secMsg_Tx(
  Cc_HANDLE *pCc,       // pointer to CANcrypt handle record
  CAN_MSG *pCANtx,      // pointer to CAN message to transmit
  CAN_MSG *pCANcpy      // Pointer to sec CAN message buffer
);


/********************************************************************
DOES:    Adds a call-back event to queue for later processing.
         Called from interrupt processes
RETURNS: nothing
********************************************************************/
void Cc_QueueEvent(
  Cc_HANDLE  *pCc,      // pointer to CANcrypt handle record
  UNSIGNED8   event,    // event that happened (Cc_EVENT_xxx)
  UNSIGNED32  param1,   // depending on the event, up to 2 
  UNSIGNED32  param2    // event related parameters
);


/********************************************************************
DOES:    Resets all timeouts used.
         Cc_TIMING_FAST            0x00
         Cc_TIMING_MEDIUM          0x01
         Cc_TIMING_SLOW            0x02

         Cc_BITMETHOD_DIRECT       0x00
         Cc_BITMETHOD_DELAY        0x40
RETURNS: nothing
********************************************************************/
void Cc_SetTimeouts(
  Cc_HANDLE *pCc,               // pointer to a CANcrypt handle record
  UNSIGNED8 timing              // timing and method used
);


/********************************************************************
DOES:    Retrieves a 32bit value from 4 bytes.
         The 4 bytes may be un-aligned.
RETURNS: The 32bit value
********************************************************************/
UNSIGNED32 mCc_Get32bit(
  UNSIGNED8 src[4]              // pointer to 4 bytes
  );


/********************************************************************
DOES:    Copies a 32bit value to 4 bytes.
         The 4 bytes may be un-aligned.
RETURNS: nothing
********************************************************************/
void mCc_Put32bit(
  UNSIGNED8 dst[4],             // pointer to 4 destination bytes
  UNSIGNED32 src                // the 32bit value
);


/********************************************************************
DOES:    Resets all message counters. Called on pairing/grouping.
RETURNS: nothing
********************************************************************/
void mCc_ResetCounters(
  Cc_HANDLE *pCc,               // pointer to a CANcrypt handle record
  UNSIGNED32 rnd                // shared random value
);


#ifdef DEBUG
/********************************************************************
CANcrypt debug output
********************************************************************/
void DebugOutCAN(
  UNSIGNED16 id, UNSIGNED8 len,
  UNSIGNED8 d0, UNSIGNED8 d1, UNSIGNED8 d2, UNSIGNED8 d3,
  UNSIGNED8 d4, UNSIGNED8 d5, UNSIGNED8 d6, UNSIGNED8 d7
  );
#endif


#ifdef CANcrypt_DEBUG_DETAIL
/********************************************************************
CANcrypt debug output, sharing key
********************************************************************/
void DebugOutKey(
  UNSIGNED16 src, 
  UNSIGNED32 *pad,
  Cc_HANDLE *cch
  );
#endif

#endif
/*----------------------- END OF FILE -----------------------------*/
