/********************************************************************
MODULE:    CANcrypt_can.h, CAN definitions and functions
CONTAINS:  CAN interface definitions for the CANcrypt system
AUTHORS:   Embedded Systems Academy, Inc (USA) and
           Embedded Systems Academy, GmbH (Germany)
HOME:      https://www.cancrypt.net
LICENSE:   See below, APPLIES TO THIS FILE ONLY. 
           See individual file headers for applicable license.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at
  www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or 
  implied. See the License for the specific language governing 
  permissions and limitations under the License.

VERSION:   1.03, 06-AUG-2018
           $LastChangedRevision: 466 $
********************************************************************/


#ifndef _CANCRYPT_CAN_H
#define _CANCRYPT_CAN_H

#include "Cc_user_types.h"
#include "can.h"

/********************************************************************
Data structure for a single CAN message 
********************************************************************/
typedef can_frame_t CAN_MSG;


/********************************************************************
DOES:    This function initializes the CAN interface.
RETURNS: 1 if init is completed 
         0 if init failed
********************************************************************/
UNSIGNED8 CCHW_Init (  
  UNSIGNED16 BaudRate   // 0 for default
  );


/********************************************************************
DOES:    Process CAN HW Rx FIFO
********************************************************************/
UNSIGNED8 CCHW_CheckRxFIFO(void);


/**************************************************************************
DOES: If there is something in the transmit queue, and if the transmit
      buffer is available, copy next message from queue to transmit buffer
***************************************************************************/ 
void CCHW_CheckTxQueue (
  UNSIGNED8 max_msg_to_process
  );


/********************************************************************
DOES:    This function implements a CAN transmit FIFO. With each
         function call a message is added to the FIFO.
RETURNS: 1 Message was added to the transmit FIFO
         0 If FIFO is full, message was not added
********************************************************************/
UNSIGNED8 CCHW_PushMessage (
  CAN_MSG MEM_FAR *pTransmitBuf // CAN message to send
  );
UNSIGNED8 CCHW_PushMessage_Internal (
  CAN_MSG MEM_FAR *pTransmitBuf // CAN message to send
  );


/********************************************************************
DOES:    This function retrieves a CAN message from the receive FIFO.
RETURNS: 1 Message was pulled from receive FIFO
         0 Queue empty, no message received
********************************************************************/
UNSIGNED8 CCHW_PullMessage (
  CAN_MSG MEM_FAR *pReceiveBuf // Buffer for received message
  );


/********************************************************************
DOES:    Transmission of a CANcrypt high priority message.
NOTE:    Must be transmit immediately, no delay. Bypasses Tx FIFO.
         ONLY REQUIRED FOR DIRECT BIT CLAIMING MODE
RETURNS: Nothing
********************************************************************/
UNSIGNED8 CCHW_TransmitNow(
  CAN_MSG *pMsg
);


/********************************************************************
DOES:    Enable/Disable of CAN receive interrupt
RETURNS: Nothing
********************************************************************/
void CCHW_EnableCANrx (void);
void CCHW_DisableCANrx (void);


#endif
/*----------------------- END OF FILE -----------------------------*/
