/********************************************************************
MODULE:    CANFIFO
CONTAINS:  CANcrypt dem,o, CAN SW Filter and FIFOs
COPYRIGHT: Embedded Systems Academy GmbH, 2016-2017
HOME:      www.esacademy.com/cancrypt
LICENSE:   LIMITED COMMERCIAL USE FOR UP TO 500 DEVICES, SEE 
           www.esacademy.com/cancrypt/license_500.txt
           FOR DETAILS!
CONTACT:   info@cancrypt.eu

Unless required by applicable law or agreed to in writing, 
software distributed under the License is distributed on an 
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
either express or implied.

VERSION:   0.900, 17-MAR-2017
********************************************************************/

#include "CANcrypt_includes.h"


#if (TXFIFOSIZE != 0) && (TXFIFOSIZE != 4) && (TXFIFOSIZE != 8) && (TXFIFOSIZE != 16) && (TXFIFOSIZE != 32) && (TXFIFOSIZE != 64)
 #error "TXFIFOSIZE must be 0 (deactivated), 4, 8, 16, 32 or 64"
#endif
#if (RXFIFOSIZE != 0) && (RXFIFOSIZE != 4) && (RXFIFOSIZE != 8) && (RXFIFOSIZE != 16) && (RXFIFOSIZE != 32) && (RXFIFOSIZE != 64) && (RXFIFOSIZE != 128)
 #error "RXFIFOSIZE must be 0 (deactivated), 4, 8, 16, 32, 64 or 128"
#endif


/**************************************************************************
GLOBAL VARIABLES
***************************************************************************/ 

typedef struct
{
#if (TXFIFOSIZE > 0)
  CAN_MSG TxFifo[TXFIFOSIZE];
#endif
#if (RXFIFOSIZE > 0)
  CAN_MSG RxFifo[RXFIFOSIZE];
#endif
#if (TXFIFOSIZE > 0)
  UNSIGNED8 TxIn;
  UNSIGNED8 TxOut;
#endif
#if (RXFIFOSIZE > 0)
  UNSIGNED8 RxIn;
  UNSIGNED8 RxOut;
#endif
} CANFIFOINFO;

// Module variable with all FIFO information
CANFIFOINFO MEM_BUF mCF;


/**************************************************************************
PUBLIC FUNCTIONS
***************************************************************************/ 


#if (TXFIFOSIZE > 0)
/**************************************************************************
DOES: Flushes / clears the TXFIFO, all data stored in FIFO is lost
***************************************************************************/ 
void CANTXFIFO_Flush (
  void
  )
{
  mCF.TxIn = 0;
  mCF.TxOut = 0;
}


/**************************************************************************
DOES:    Returns a CAN message pointer to the next free location in FIFO.
         Application can then copy a CAN message to the location given by 
         the pointer and MUST call CANTXFIFO_InDone() when done.
RETURNS: CAN message pointer into FIFO
         NULL if FIFO is full
***************************************************************************/ 
CAN_MSG MEM_BUF *CANTXFIFO_GetInPtr (
  void
  )
{
UNSIGNED8 ovr; // check if FIFO is full

  ovr = mCF.TxIn + 1;
  ovr &= (TXFIFOSIZE-1);

  if (ovr != mCF.TxOut)
  {// FIFO is not full
    return &(mCF.TxFifo[mCF.TxIn]);
  }
  return 0;
}


/**************************************************************************
DOES:    Must be called by application after data was copied into the FIFO,
         this increments the internal IN pointer to the next free location 
         in the FIFO.
RETURNS: nothing
***************************************************************************/ 
void CANTXFIFO_InDone (
  void
  )
{
  // Increment IN pointer
  mCF.TxIn++;
  mCF.TxIn &= (TXFIFOSIZE-1);
}


/**************************************************************************
DOES:    Returns a CAN message pointer to the next OUT message in the FIFO.
         Application can then copy the CAN message from the location given 
         by the pointer to the desired destination and MUST call 
         CANTXFIFO_OutDone() when done.
RETURNS: CAN message pointer into FIFO      
         NULL if FIFO is empty
***************************************************************************/ 
CAN_MSG MEM_BUF *CANTXFIFO_GetOutPtr (
  void
  )
{
  if (mCF.TxIn != mCF.TxOut)
  { // message available in FIFO
    return &(mCF.TxFifo[mCF.TxOut]);
  }
  return 0;
}


/**************************************************************************
DOES:    Must be called by application after data was copied from the FIFO,
         this increments the internal OUT pointer to the next location 
         in the FIFO.
RETURNS: nothing
***************************************************************************/ 
void CANTXFIFO_OutDone (
  void
  )
{
  mCF.TxOut++;
  mCF.TxOut &= (TXFIFOSIZE-1);
}
#endif // (TXFIFOSIZE > 0)


#if (RXFIFOSIZE > 0)
/**************************************************************************
DOES: Flushes / clears the RXFIFO, all data stored in FIFO is lost
***************************************************************************/ 
void CANRXFIFO_Flush (
  void
  )
{
  mCF.RxIn = 0;
  mCF.RxOut = 0;
}


/**************************************************************************
DOES:    Returns a CAN message pointer to the next free location in FIFO.
         Application can then copy a CAN message to the location given by 
         the pointer and MUST call CANRXFIFO_InDone() when done.
RETURNS: CAN message pointer into FIFO      
         NULL if FIFO is full
***************************************************************************/ 
CAN_MSG MEM_BUF *CANRXFIFO_GetInPtr (
  void
  )
{
UNSIGNED8 ovr; // check if FIFO is full

  ovr = mCF.RxIn + 1;
  ovr &= (RXFIFOSIZE-1);

  if (ovr != mCF.RxOut)
  {// FIFO is not full
    return &(mCF.RxFifo[mCF.RxIn]);
  }
  return 0;
}


/**************************************************************************
DOES:    Must be called by application after data was copied into the FIFO,
         this increments the internal IN pointer to the next free location 
         in the FIFO.
RETURNS: nothing
***************************************************************************/ 
void CANRXFIFO_InDone (
  void
  )
{
  // Increment IN pointer
  mCF.RxIn++;
  mCF.RxIn &= (RXFIFOSIZE-1);
}


/**************************************************************************
DOES:    Returns a CAN message pointer to the next OUT message in the FIFO.
         Application can then copy the CAN message from the location given 
         by the pointer to the desired destination and MUST call 
         CANRXFIFO_OutDone() when done.
RETURNS: CAN message pointer into FIFO      
         NULL if FIFO is empty
***************************************************************************/ 
CAN_MSG MEM_BUF *CANRXFIFO_GetOutPtr (
  void
  )
{
  if (mCF.RxIn != mCF.RxOut)
  { // message available in FIFO
    return &(mCF.RxFifo[mCF.RxOut]);
  }
  return 0;
}


/**************************************************************************
DOES:    Must be called by application after data was copied from the FIFO,
         this increments the internal OUT pointer to the next location 
         in the FIFO.
RETURNS: nothing
***************************************************************************/ 
void CANRXFIFO_OutDone (
  void
  )
{
  mCF.RxOut++;
  mCF.RxOut &= (RXFIFOSIZE-1);
}
#endif // (RXFIFOSIZE > 0)


/**************************************************************************
END OF FILE
**************************************************************************/

