/********************************************************************
MODULE:    MAIN
CONTAINS:  CANcrypt FD demo, see readme.txt for more info
COPYRIGHT: 2016-2018 Embedded Systems Academy, Inc (USA) and
           Embedded Systems Academy, GmbH (Germany)
HOME:      https://www.cancrypt.net
LICENSE:   The CANcrypt software in this distribution may be used 
           freely on NXP LPC546xx devices. When used on other devices
           or hardware, a commercial license is required, contact us
           at https://www.cancrypt.net for more information.

Unless required by applicable law or agreed to in writing, 
software distributed under the License is distributed on an 
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
either express or implied.

VERSION:   0.1, 06-AUG-2018
           $LastChangedRevision: 466 $
********************************************************************/

#include "CANcrypt_includes.h"
#include "board.h"
#include "pin_mux.h"


/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

// CANcrypt handle
Cc_HANDLE CcH;

// Counter for procss control
UNSIGNED32 counter = 0;

// grouping status
UNSIGNED8 grouped = FALSE;

// secure message transmit repetiton time
CC_TIMER_TYPE msg_time;

// timer for on off handling
UNSIGNED16 onoff_time;

// Counter for secure heartbeats
UNSIGNED32 secHBcnt = 0;

// Device identification, 4 * 32bit
UNSIGNED32 my_ident[4] = {0xC0D1F1EDul, 0x00000100ul, 0x00010001ul, 0x00001000ul + Cc_DEVICE_ID};

#ifdef Cc_USE_SECURE_MSG
/********************************************************************
Secure messages used
********************************************************************/
// CAN ID, first byte encrypted, number of bytes encrypted,
// methods used, device address of producer
//#define Cc_secMSG_2(prd)        {0x0180 + prd, COBID_MASKID, 0, 0, Cc_METHOD, prd}
//#define Cc_secMSG_3(prd)        {0x0180 + prd, COBID_MASKID, 0, 0, Cc_METHOD, prd}
//#define Cc_secMSG_7(prd)        {0x0180 + prd, COBID_MASKID, 0, 0, Cc_METHOD, prd}
#define Cc_secMSG_2(prd)        {0x0180 + prd, 0x01FFFF5FF,  2, 3, Cc_METHOD, prd} // covers ID 0x18X and 0x38X
#define Cc_secMSG_3(prd)        {0x0180 + prd, COBID_MASKID, 5, 5, Cc_METHOD, prd}
#define Cc_secMSG_7(prd)        {0x0180 + prd, COBID_MASKID, 0,56, Cc_METHOD, prd}
#define Cc_secMSG_last          {COBID_MASKID,COBID_MASKID,0xFF,0xFF,0xFF,0xFF}

// secure transmit message table
Cc_SEC_MSG_TABLE_ENTRY TxSecMg[2] = {
#if (Cc_DEVICE_ID == 2)
                                     Cc_secMSG_2(Cc_DEVICE_ID),
#endif
#if (Cc_DEVICE_ID == 3)
                                     Cc_secMSG_3(Cc_DEVICE_ID),
#endif
#if (Cc_DEVICE_ID == 7)
                                     Cc_secMSG_7(Cc_DEVICE_ID),
#endif
                                     Cc_secMSG_last};
UNSIGNED16 TxTrk[1];

// secure receive message table
Cc_SEC_MSG_TABLE_ENTRY RxSecMg[3] = {
#if (Cc_DEVICE_ID != 2)
                                     Cc_secMSG_2(2),
#endif
#if (Cc_DEVICE_ID != 3)
                                     Cc_secMSG_3(3),
#endif
#if (Cc_DEVICE_ID != 7)
                                     Cc_secMSG_7(7),
#endif
                                     Cc_secMSG_last};
UNSIGNED16 RxTrk[3];
#endif


/*****************************************************************************
DOES:    Hard fault handler, indicate state by by LED 3
*****************************************************************************/
void HardFault_Handler(void)
{
#ifdef Cc_USE_DIGOUT
  Cc_DIGOUT1_ON();
  Cc_DIGOUT2_OFF();
  Cc_DIGOUT3_ON();
  while(1);
#endif
}


/*****************************************************************************
DOES:    Transmits a CANopen style heartbeat
RETURNS: Nothing
*****************************************************************************/
void APP_TxHB (UNSIGNED8 status)
{
  CAN_MSG can;
  
  can.id = 0x700 + Cc_DEVICE_ID;
  can.dataByte[0] = status;
  can.length = 1;
  can.format = kCAN_FrameFormatStandard;
  can.type = kCAN_FrameTypeData;
  can.proto = kCAN_ProtoTypeFD;
  can.bitratemode = kCAN_BitrateModeTypeSwitch;
  CCHW_PushMessage(&can);
}


/*****************************************************************************
DOES:    Sends a CANopen style emergency message
RETURNS: Nothing
*****************************************************************************/
void APP_TxEMCY(
  UNSIGNED16 err,
  UNSIGNED8 d0, UNSIGNED8 d1, UNSIGNED8 d2, UNSIGNED8 d3, UNSIGNED8 d4
  )
{
CAN_MSG emcy;

  emcy.id = 0x80+Cc_DEVICE_ID;
  emcy.dataByte[0] = err;
  emcy.dataByte[1] = err >> 8;
  emcy.dataByte[2] = 1;
  emcy.dataByte[3] = d0;
  emcy.dataByte[4] = d1;
  emcy.dataByte[5] = d2;
  emcy.dataByte[6] = d3;
  emcy.dataByte[7] = d4;
  emcy.length = 8;
  emcy.format = kCAN_FrameFormatStandard;
  emcy.type = kCAN_FrameTypeData;
  emcy.proto = kCAN_ProtoTypeFD;
  emcy.bitratemode = kCAN_BitrateModeTypeSwitch;

  CCHW_PushMessage(&emcy);
}


/*****************************************************************************
DOES:    Trigger next transmit of a secure message
RETURNS: Nothing
*****************************************************************************/
void TriggerSecureTx()
{
  static UNSIGNED8 cnt;
  CAN_MSG tx;
  
  tx.dataByte[0] = 0x11;
  tx.dataByte[1] = Cc_DEVICE_ID;
  tx.dataByte[2] = cnt;
  tx.dataByte[3] = Cc_DEVICE_ID;
  tx.dataByte[4] = 0xAA;
  memcpy(&(tx.dataByte[5]),"Plain text part, not changing.",30);
#if (Cc_DEVICE_ID == 2)
  tx.length = 5;
#endif
#if (Cc_DEVICE_ID == 3)
  tx.length = 10;
#endif
#if (Cc_DEVICE_ID == 7)
  tx.length = 3 + (cnt & 0x1F);
#endif
  tx.format = kCAN_FrameFormatStandard;
  tx.type = kCAN_FrameTypeData;
  tx.proto = kCAN_ProtoTypeFD;
  tx.bitratemode = kCAN_BitrateModeTypeSwitch;
#if 0
 #warning "SHARING SECRETS is active: show plaintext before transmit"
  // Demo only, transmit original first
  tx.id = 0x480 + Cc_DEVICE_ID;
  CCHW_PushMessage(&tx);
#endif
  // Transmit secure (recognized by ID)
  tx.id = 0x180 + Cc_DEVICE_ID;
  CCHW_PushMessage(&tx);
  cnt++;
#if (Cc_DEVICE_ID == 2)
  if (cnt & 1)
  { // on every second call, transmit a second message
    tx.dataByte[0] = 0x33;
    tx.dataByte[4] = 0xCC;
    tx.length = 8;
    tx.id = 0x380 + Cc_DEVICE_ID;
    CCHW_PushMessage(&tx);
  }
#endif
}


/*****************************************************************************
DOES:    Reports a CANcrypt system event to the application
RETURNS: Nothing
*****************************************************************************/
void Cccb_Event(
  UNSIGNED8 event,              // Cc_EVENT_xxx
  UNSIGNED32 param1,            // see Cc_EVENT_xxx definitions
  UNSIGNED32 param2
)
{
  switch (event)
  {
    case Cc_EVENT_GROUP_INIT: // send CANopen HB, pre-operational
      APP_TxHB(0x7F);
      grouped = 1;
      break;
    
    case Cc_ERR_GROUP_KEYID: // different key requested, here allow user key
      if ((param1 & 0xF) == Cc_PERM_KEY_USER)
      { // user key requested
        if (!(Cc_SelectGroup(&CcH,Cc_PERM_KEY_USER,Cc_PERMKEY_LEN32,0x008C,0x008C))) // 2,3,7
        { // Session key not yet available, generate first
          Cc_SelectGroup(&CcH,Cc_PERM_KEY_USER+Cc_GRPKEY_GENERATE,Cc_PERMKEY_LEN32,0x008C,0x008C); // 2,3,7
        }
        grouped = 0;
        // start again
        counter = 0;
      }
      else
      { // Requested key not supported
        Cc_TxAlert(&CcH,0,(((UNSIGNED16)event) << 8) + (param1 & 0xFF));
      }
      break;
    case Cc_ERR_GROUP_KEYFAIL: // key did not work, here go back to user key
      Cc_TxAlert(&CcH,0,(((UNSIGNED16)event) << 8) + (param1 & 0xFF));
      if (param1 == Cc_PERM_KEY_SESSION)
      { // last key used was session key, try again with user key
        Cc_SelectGroup(&CcH,Cc_PERM_KEY_USER,Cc_PERMKEY_LEN32,0x008C,0x008C); // 2,3,7
      }
      grouped = 0;
      // start again
      counter = 0;
      break;
    case Cc_ERR_GROUP_KEYGEN: // informational: key generation started
    case Cc_ERR_GROUP_KEYGEN_REQ:
      Cc_TxAlert(&CcH,0,(((UNSIGNED16)event) << 8) + (param1 & 0xFF));
      break;
    case Cc_ERR_GROUP_KEYNOGEN:
    case Cc_ERR_KEY_NO_KEY: // requested key not available in NVOL
      Cc_TxAlert(&CcH,0,(((UNSIGNED16)event) << 8) + (param1 & 0xFF));
      Cc_SelectGroup(&CcH,Cc_PERM_KEY_USER,Cc_PERMKEY_LEN32,0x008C,0x008C); // 2,3,7
      onoff_time = CCHW_GetTime() + 10000; // wait max of 10s in this state
      grouped = 0;
      // start again
      counter = 0;
      break;
      
    case Cc_EVENT_GROUPED: // send CANopen HB, operational
      APP_TxHB(0x05);
      grouped = 2;
      // initial repetiton time for secure transmit
      msg_time = CCHW_GetTime() + 1000;
      onoff_time = CCHW_GetTime() + 20000 + (CCHW_Rand() & 0x1FFF); // 20 to 38s
      break;

    case Cc_EVENT_UNGROUP_SAVE:
    case Cc_EVENT_UNPAIR: // send CANopen HB, stop
      APP_TxHB(0x04);
      grouped = 0;
      // start again
      counter = 0;
      // on next start use session key
      Cc_SelectGroup(&CcH,Cc_PERM_KEY_SESSION,Cc_PERMKEY_LEN32,0x008C,0x008C); // 2,3,7
      break;
    
    case Cc_ERR_SECHB_FAIL:
      APP_TxEMCY(0x9388,param1,0,0,0,0);
      grouped = 0;
      // start again
      counter = 0;
      break;
    case Cc_ERR_GROUP_LOST:
      APP_TxEMCY(0x9399,param1,param1>>8,param2,param2>>8,0);
      grouped = 0;
      // start again
      counter = 0;
      break;
    case Cc_ERR_MSG_NOSEC:
      APP_TxEMCY(0x93AA,param1,param1>>8,param2,param2>>8,0);
      break;
    case Cc_ERR_MSG_UNEXPECT: // send CANopen emergency
      APP_TxEMCY(0x93BB,param1,param1>>8,param2,param2>>8,0);
      break;
    
    case Cc_EVENT_HBSECURED:
      if (grouped == 2)
      {
        APP_TxHB(0x05);
      }
      else
      {
        APP_TxHB(0x7F);
      }
      secHBcnt++;
      break;
    default:
      APP_TxEMCY(0x0011,event,param1,param1>>8,param2,param2>>8);
      if (event >= 0x80)
      { // error
        grouped = 0;
        // start again
        counter = 0;
      }
      break;
  }
}


/*******************************************************************************
 * MAIN
 ******************************************************************************/

/*!
 * @brief Main function
 */
int main(void)
{
CAN_MSG CANapp;

#ifdef Cc_USE_DIGOUT
  /* Define the init structure for the output LED pin*/
  gpio_pin_config_t led_config = {
      kGPIO_DigitalOutput, 0,
  };
#endif

  /* Board pin, clock, debug console init */
  /* attach 12 MHz clock to FLEXCOMM0 (debug console) */
  CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);
  CLOCK_EnableClock(kCLOCK_Gpio0);
  CLOCK_EnableClock(kCLOCK_Gpio1);
  CLOCK_EnableClock(kCLOCK_Gpio2);
  CLOCK_EnableClock(kCLOCK_Gpio3);

  BOARD_InitPins();
  BOARD_BootClockFROHF48M();
  BOARD_InitDebugConsole();
 
#ifdef Cc_USE_DIGOUT
  /* Init output LED GPIO. */
  GPIO_PinInit(GPIO, BOARD_LED1_GPIO_PORT, BOARD_LED1_GPIO_PIN, &led_config);
  GPIO_PinInit(GPIO, BOARD_LED2_GPIO_PORT, BOARD_LED2_GPIO_PIN, &led_config);
  GPIO_PinInit(GPIO, BOARD_LED3_GPIO_PORT, BOARD_LED3_GPIO_PIN, &led_config);
#endif

  /* configure for 4Mbps data 1Mbps nominal, CAN-FD */
  CCHW_Init(0);
    
  // CANcrypt default key selection
  if (!(Cc_SelectGroup(&CcH,Cc_PERM_KEY_SESSION,Cc_PERMKEY_LEN32,0x008C,0x008C))) // 2,3,7
  { // Session key not yet available, generate first
    Cc_SelectGroup(&CcH,Cc_PERM_KEY_SESSION+Cc_GRPKEY_GENERATE,Cc_PERMKEY_LEN32,0x008C,0x008C); // 2,3,7
  }

  // foreground loop
  while(1)
  {
  
    // Work on CAN HW receive and transmit FIFO
    // In this version nothing processed in interrupt !!!
    CCHW_CheckTxQueue(4);
    if (CCHW_CheckRxFIFO() == 2)
    { // overrun
      APP_TxEMCY(0xEE01,'R','X','O','V','R');
    }

    if (counter == 0)
    {
      // All LEDs on
#ifdef Cc_USE_DIGOUT
      Cc_DIGOUT1_ON();
      Cc_DIGOUT2_ON();
      Cc_DIGOUT3_ON();
#endif
      // send boot up message
      APP_TxHB(0x00);
    }
    
    // use counter as control as when to enable CANcrypt
    if (counter == 1000)
    {
      // All LEDs off
#ifdef Cc_USE_DIGOUT
      Cc_DIGOUT1_OFF();
      Cc_DIGOUT2_OFF();
      Cc_DIGOUT3_OFF();
#endif
      Cc_Restart(&CcH,Cc_DEVICE_ID,Cc_GROUP_CTRL_RESTART,Cccb_Event,CCHW_PushMessage,NULL,my_ident);
#ifdef Cc_USE_SECURE_MSG
      Cc_Load_Sec_Msg_Table(&CcH,RxSecMg,RxTrk,TxSecMg,TxTrk);
#endif
      // on / off timing
      onoff_time = CCHW_GetTime() + 10000;
      // initial repetion time for secure transmit
      msg_time = CCHW_GetTime() + 1000 + (250*Cc_DEVICE_ID);
    }

    if (counter >= 1000)
    {
      Cc_Process_Tick(&CcH);
      
      if (grouped == 0)
      {
        if (CCHW_IsTimeExpired(onoff_time))
        { // start over
          counter = 0;
        }
      }
      else if (grouped == 2)
      { // cyclic secure tranmit
        if (CCHW_IsTimeExpired(msg_time))
        {
          TriggerSecureTx();
          // next trigger ranges randomly from 100ms to about 350ms
          msg_time = CCHW_GetTime() + 100 + (CCHW_Rand() & 0xFF);
        }
        // shut off
        if (CCHW_IsTimeExpired(onoff_time))
        {
          Cc_TxDisconnect(&CcH,0,Cc_EVENT_UNGROUP_SAVE);
          onoff_time += 3000; // do not come back for 3s
        }
      }
    }
   
    // stay here
    if (counter < 10000)
    {
      counter++;
    }
 
    // Always process incoming CAN messages for application
    if (CCHW_PullMessage(&CANapp))
    {
      if ((CANapp.id == 0) && (CANapp.dataByte[0] == 0x81))
      { // CANopen NMT Device Reset, here start over, with erase of user key
        Ccnvol_ErasePermKey(Cc_PERM_KEY_USER);
        counter = 0;
        grouped = 0;
      }
      else if ((CANapp.id == 0) && (CANapp.dataByte[0] == 0x82))
      { // CANopen NMT Communication Reset, here start over
        counter = 0;
        grouped = 0;
      }
#warning "SHARING SECRETS is active: show plaintext after receive"
      else if ((CANapp.id & 0x07F0) == 0x180)
      { // secure data received, transmit echo
        CANapp.id &= 0x000F;
        CANapp.id += 0x200;
        CANapp.id += (Cc_DEVICE_ID << 4);
        CCHW_PushMessage(&CANapp);
      }
      else if ((CANapp.id & 0x07F0) == 0x380)
      { // secure data received, transmit echo
        CANapp.id &= 0x000F;
        CANapp.id += 0x400;
        CANapp.id += (Cc_DEVICE_ID << 4);
        CCHW_PushMessage(&CANapp);
      }
    }

  } // end of while(1)
} // end of main
