/********************************************************************
MODULE:    NVOL_Keys, key storage
CONTAINS:  Here: only a simulation of non-volatile key storage, using
           hard coded const code space and RAM for storage.
COPYRIGHT: 2016-2018 Embedded Systems Academy, Inc (USA) and
           Embedded Systems Academy, GmbH (Germany)
HOME:      www.cancrypt.eu
LICENSE:   LIMITED COMMERCIAL USE. FOR DETAILS SEE 
           www.cancrypt.eu/docs/CANcryptLicense.pdf

Unless required by applicable law or agreed to in writing, 
software distributed under the License is distributed on an 
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
either express or implied.

VERSION:   1.03, 14-AUG-2018
           $LastChangedRevision: 519 $
********************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

#include <string.h>
#include "CANcrypt_includes.h"



/********************************************************************
MODULE VARIABLES
Here all key data stored in RAM, only for demo. Move to NVOL.
********************************************************************/ 

// Bit 0x10: ID stored, bit 0x01: key stored
UNSIGNED8 key_status[5] = {0,0,0,0,0};
// here simulation, all keys are only stored in RAM
// Replace with NVOL storage in real driver / application
UNSIGNED32 keys_all_RAM[5][Cc_PERMKEY_LEN32];
UNSIGNED32 keyIDs_all_RAM[5];

// permanent key currently in use
UNSIGNED32 key_in_use[Cc_PERMKEY_LEN32];

// Device identification, 4 * 32bit
extern UNSIGNED32 my_ident[4];

/********************************************************************
LOCAL FUNCTIONS
********************************************************************/

/********************************************************************
DOES:    Encrypts a key before storage.
NOTE:    Encryption should be based on a chip/device specific key,
         like a PUF key or a key stored ina  secure area or at least
         related to the serial number of the device.
RETURNS: Nothing
********************************************************************/
static void EncryptKey(
  UNSIGNED32 *pKdst,
  UNSIGNED32 *pKsrc
)
{
UNSIGNED8 lp;
UNSIGNED32 device_key[Cc_PERMKEY_LEN32];
UNSIGNED32 use_key[Cc_KEY_LEN32];
  
#if ((Cc_PERMKEY_LEN32 / 2) != Cc_KEY_LEN32)
  #error "This version requires length permanent key = 2 * length dyn key"
#endif
  
  // If no device specific key is available, generate by serial number
  device_key[0] = 0xDEADC0DE + my_ident[3];
  for (lp = 1; lp < Cc_PERMKEY_LEN32; lp++)
  { // generate some default key based on serial number
    device_key[lp] = (device_key[lp-1] + 0xBADC0FFE) ^ (my_ident[lp&3] + lp);
  }
  
  memcpy(pKdst,pKsrc,Cc_PERMKEY_LEN8);
  // Generate 1st key
  Ccuser_MakeKey(device_key,&(device_key[Cc_KEY_LEN32]),my_ident[3],use_key);
  // Encrypt 1st half of perm key
  Ccuser_Encrypt(use_key,pKdst,0,Cc_KEY_LEN8);
  // Generate 2nd key
  Ccuser_MakeKey(&(device_key[Cc_KEY_LEN32]),use_key,my_ident[3],use_key);
  // Encrypt 2nd half of perm key
  Ccuser_Encrypt(use_key,&(pKdst[Cc_KEY_LEN32/2]),0,Cc_KEY_LEN8);
}


/********************************************************************
DOES:    Decrypts a key before storage.
RETURNS: Nothing
********************************************************************/
static void DecryptKey(
  UNSIGNED32 *pKdst,
  UNSIGNED32 *pKsrc
)
{
UNSIGNED8 lp;
UNSIGNED32 device_key[Cc_PERMKEY_LEN32];
UNSIGNED32 use_key[Cc_KEY_LEN32];
  
#if ((Cc_PERMKEY_LEN32 / 2) != Cc_KEY_LEN32)
  #error "This version requires length permanent key = 2 * length dyn key"
#endif
  
  // If no device specific key is available, generate by serial number
  device_key[0] = 0xDEADC0DE + my_ident[3];
  for (lp = 1; lp < Cc_PERMKEY_LEN32; lp++)
  { // generate some default key based on serial number
    device_key[lp] = (device_key[lp-1] + 0xBADC0FFE) ^ (my_ident[lp&3] + lp);
  }
  
  memcpy(pKdst,pKsrc,Cc_PERMKEY_LEN8);
  // Generate 1st key
  Ccuser_MakeKey(device_key,&(device_key[Cc_KEY_LEN32]),my_ident[3],use_key);
  // Encrypt 1st half of perm key
  Ccuser_Decrypt(use_key,pKdst,0,Cc_KEY_LEN8);
  // Generate 2nd key
  Ccuser_MakeKey(&(device_key[Cc_KEY_LEN32]),use_key,my_ident[3],use_key);
  // Encrypt 2nd half of perm key
  Ccuser_Decrypt(use_key,&(pKdst[Cc_KEY_LEN32/2]),0,Cc_KEY_LEN8);
}


/********************************************************************
PUBLIC FUNCTIONS
********************************************************************/

/********************************************************************
BOOK:    Section 6.1 "Key hierarchy access"
DOES:    This function directly returns a key from the key hierarchy
RETURNS: Pointer to the key or NULL if not available
********************************************************************/
UNSIGNED32 *Ccnvol_GetPermKey(
  UNSIGNED8 key_major_ID       // key major ID, 2 to 6
)
{
  UNSIGNED32 *pKey = NULL;
  
  if ( (key_major_ID >= 2) && (key_major_ID < 6) )
  { // range check
    if (key_status[key_major_ID-2] == 0x11)
    { // key available
      pKey = &(key_in_use[0]);
      DecryptKey(pKey,&(keys_all_RAM[key_major_ID-2][0]));
      
#ifdef CANcrypt_DEBUG
      DebugOutCAN(Cc_DEVICE_ID+0x100,2,0x64,key_major_ID,0,0,0,0,0,0);
      DebugOutCAN(Cc_DEVICE_ID+0x100,8,pKey[0],pKey[0]>>8,pKey[0]>>16,pKey[0]>>24,pKey[1],pKey[1]>>8,pKey[1]>>16,pKey[1]>>24);
      //DebugOutCAN(Cc_DEVICE_ID+0x100,8,pKey[2],pKey[2]>>8,pKey[2]>>16,pKey[2]>>24,pKey[3],pKey[3]>>8,pKey[3]>>16,pKey[3]>>24);
#endif
      
    }
  }
#if 0 // (Cc_DEVICE_ID == 7)
  if (key_major_ID == Cc_PERM_KEY_SESSION)
  {
    // For test, introduce error in last saved session key, flip one bit
    session_key[0] ^= 1;
  }
#endif
  
  return pKey;
}


/**************************************************************************
BOOK:    Section 6.1 "Key hierarchy access"
DOES:    This function returns a key identifier from the key hierarchy
RETURNS: 0 if not available, else 32bit unique key identifier
**************************************************************************/
UNSIGNED32 Ccnvol_GetPermKeyID(
  UNSIGNED8 key_major_ID      // key major ID
)
{
  UNSIGNED32 ret_val = 0;
  
  if ( (key_major_ID >= Cc_PERM_KEY_SESSION) && (key_major_ID <= Cc_PERM_KEY_INTEGRATOR) )
  { // range check
    if (key_status[key_major_ID-2] == 0x11)
    {
      ret_val = keyIDs_all_RAM[key_major_ID-2];
    }
  }
  
  return ret_val;
}


/**************************************************************************
BOOK:    Section 6.1 "Key hierarchy access"
DOES:    This function writes a key identifier into the key hierarchy
RETURNS: TRUE when saved, else FALSE
**************************************************************************/
UNSIGNED8 Ccnvol_SavePermKeyID(
  UNSIGNED8 key_major_ID,       // key major ID
  UNSIGNED32 key_unique_ID      // unique 32bit key ID
)
{
  UNSIGNED32 ret_val = 0;
  
#ifdef CANcrypt_DEBUG
      DebugOutCAN(Cc_DEVICE_ID+0x100,4,key_unique_ID,key_unique_ID>>8,key_unique_ID>>16,key_unique_ID>>24,0,0,0,0);
#endif

  if ( (key_major_ID >= Cc_PERM_KEY_SESSION) && (key_major_ID <= Cc_PERM_KEY_INTEGRATOR) && ((key_status[key_major_ID-2] & 0x10) == 0))
  { // range check, do not allow overwrite, once saved
    keyIDs_all_RAM[key_major_ID-2] = key_unique_ID;
    key_status[key_major_ID-2] |= 0x10;
    ret_val = TRUE;
  }
  
  return ret_val;
}


/********************************************************************
BOOK:    Section 6.1 "Key hierarchy access"
DOES:    This function erases a key from the key hierarchy. Will only
         be called from CANcrypt, if called from authorized 
         configurator.
RETURNS: TRUE, if key was erased, else FALSE
********************************************************************/
UNSIGNED8 Ccnvol_ErasePermKey(
  UNSIGNED8 key_major_ID        // key major ID, 2 to 6
)
{
  UNSIGNED32 ret_val = FALSE;
  
  if ( (key_major_ID >= Cc_PERM_KEY_SESSION) && (key_major_ID <= Cc_PERM_KEY_INTEGRATOR) )
  { // range check
    memset(&(keys_all_RAM[key_major_ID-2][0]),0,Cc_PERMKEY_LEN8);
    keyIDs_all_RAM[key_major_ID-2] = 0;
    key_status[key_major_ID-2] = 0;
    ret_val = TRUE;
  }
  
  return ret_val;
}


/********************************************************************
BOOK:    Section 6.1 "Key hierarchy access"
DOES:    This function saves a key to the key hierarchy. Will only
         be called from CANcrypt, if called from authorized 
         configurator.
RETURNS: TRUE, if key was erased, else FALSE
********************************************************************/
UNSIGNED8 Ccnvol_SavePermKey(
  UNSIGNED8 key_major_ID,       // key major ID, 2 to 6
  UNSIGNED32 *pKey              // pointer to key data
)
{
  UNSIGNED32 ret_val = FALSE;
  
#ifdef CANcrypt_DEBUG
  DebugOutCAN(Cc_DEVICE_ID+0x100,2,0x65,key_major_ID,0,0,0,0,0,0);
  DebugOutCAN(Cc_DEVICE_ID+0x100,8,pKey[0],pKey[0]>>8,pKey[0]>>16,pKey[0]>>24,pKey[1],pKey[1]>>8,pKey[1]>>16,pKey[1]>>24);
  //DebugOutCAN(Cc_DEVICE_ID+0x100,8,pKey[2],pKey[2]>>8,pKey[2]>>16,pKey[2]>>24,pKey[3],pKey[3]>>8,pKey[3]>>16,pKey[3]>>24);
#endif

  if ( (key_major_ID >= Cc_PERM_KEY_SESSION) && (key_major_ID <= Cc_PERM_KEY_INTEGRATOR) && ((key_status[key_major_ID-2] & 0x01) == 0))
  { // range check, do not allow overwrite, once saved
    EncryptKey(&(keys_all_RAM[key_major_ID-2][0]),pKey);
    key_status[key_major_ID-2] |= 0x01;
    ret_val = TRUE;
  }
  
  return ret_val;
}

#ifdef __cplusplus
}
#endif

/*----------------------- END OF FILE -----------------------------*/
