/********************************************************************
MODULE:    Cc_user_default.c, CANcrypt Customizable Functions Default
CONTAINS:  The default set of customizable functions for CANcrypt
COPYRIGHT: 2016-2018 Embedded Systems Academy, Inc (USA) and
           Embedded Systems Academy, GmbH (Germany)
HOME:      https://www.cancrypt.net
LICENSE:   The CANcrypt software in this distribution may be used 
           freely on NXP LPC546xx devices. When used on other devices
           or hardware, a commercial license is required, contact us
           at https://www.cancrypt.net for more information.

Unless required by applicable law or agreed to in writing, 
software distributed under the License is distributed on an 
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
either express or implied.

VERSION:   0.1, 06-AUG-2018
           $LastChangedRevision: 466 $
********************************************************************/


#include <string.h>

#ifdef __cplusplus
extern "C" {
#endif

#include "CANcrypt_includes.h"

#ifdef Cc_USE_DIGOUT
  #include "board.h"
  #include "pin_mux.h"
#endif

#if (Cc_FUNCTIONALITY == Cc_SECFCT_ADVANCED)
 #include "../mbedtls/aes.h"
#endif

#ifdef CANcrypt_DEBUG
  // CANcrypt handle
extern Cc_HANDLE CcH;
#endif

#ifdef CANcrypt_DEBUG_DETAIL
 #ifndef _MSC_VER
 #warning "DEBUG ACTIVE: sharing pads and keys"
 #endif
#endif

  
/********************************************************************
This file implements the CANcrypt customizable security functions for
the security modes basic, regular and advanced. 
********************************************************************/
#if (Cc_FUNCTIONALITY >= Cc_SECFCT_CUSTOM)
 #error "Cc_FUNCTIONALITY must be basic, regular or advanced!"
#endif
#if (((Cc_VERSION_NR & 0x3F) != Cc_VERSION_NR) || ((Cc_REVISION_NR & 0x3F) != Cc_REVISION_NR))
 #error "Version/revision must each be in range of 0 to 63"
#endif
#if (Cc_VERSION_NR != 0)
 #error "File does not implement version 0 of CANcrypt default functions"
#endif
#if (Cc_REVISION_NR != 1)
 #error "File does not implement revision 1 of CANcrypt default functions"
#endif
#if ((Cc_PERMKEY_LEN_BITS != 128) && (Cc_PERMKEY_LEN_BITS != 256) && (Cc_PERMKEY_LEN_BITS != 512))
 #warning "This module only tested for permanent key length of 128, 256 or 512 bit!"
#endif
#if ((Cc_KEY_LEN_BITS != 64) && (Cc_KEY_LEN_BITS != 128))
 #warning "This module only tested for a dynamic key length of 64 or 128 bit!"
#endif
#if (Cc_KEY_LEN_BITS > Cc_PERMKEY_LEN_BITS)
 #error "This version requires len Cc_PERMKEY_LEN_BITS >= Cc_KEY_LEN_BITS"
#endif
#if (Cc_FUNCTIONALITY == Cc_SECFCT_ADVANCED)
 #if (Cc_KEY_LEN_BITS != 128)
  #error "Unsupported dynamic key length, must be 128 in advanced mode"
 #endif
#endif


/********************************************************************
Selecting ciphers and number of rounds used based on Cc_FUNCTIONALITY
NOTE: recommended rounds for full protection are
      SPECK32  22 rounds with 64bit key
               here Mix32 is never used "alone", just to
               create further device specific variations of keys,
               also usable with 128bit key
      SPECK64: 27 rounds with 128bit key
               here also usable with 256bit key
      XTEA64:  32 rounds (64 total, two per loop) with 128bit key
               Considered safe with > 36 total
               here also usable with 256bit key
All known attack vectors for ciphers above require the same key
to be used on a lot of data. Here keys change frequently and are
only used for small amounts of data.
********************************************************************/
#if (Cc_FUNCTIONALITY == Cc_SECFCT_BASIC)
  #define Ccuser_Mix64(a,b,c,d)   Ccuser_Mix64_SPECK(a,b,c,d)
  #define Cc_MIX64_ROUNDS         27
#elif (Cc_FUNCTIONALITY == Cc_SECFCT_REGULAR)
  #define Ccuser_Mix64(a,b,c,d)   Ccuser_Mix64_XTEA(a,b,c,d)
  #define Cc_MIX64_ROUNDS         19
#elif (Cc_FUNCTIONALITY == Cc_SECFCT_ADVANCED)
  #define Ccuser_Mix64(a,b,c,d)   Ccuser_Mix64_XTEA(a,b,c,d)
  #define Cc_MIX64_ROUNDS         31
#else
  #error "Custom Cc_FUNCTIONALITY not implemented by this module"
#endif


/********************************************************************
Macros to rotate 32bit value right or left and a single mix up round
in add-rotate-xor (ARX) style as used by Speck cipher
********************************************************************/
#define ROR32(x,r) ( (x >> (r & 0x1F)) | (x << (32 - (r & 0x1F))) )
#define ROL32(x,l) ( (x << (l & 0x1F)) | (x >> (32 - (l & 0x1F))) )
#define MIXROUND32(a,b,k)  (a=ROR32(a,8),a+=b,a^=k,b=ROL32(b,3),b^=a)
// extended CANcrypt version, create a disturbance in algorithm to make it less predictable
// if you want to make use of such a feature, we recommend to use your own, secret distortion
#define MIXROUND32x(a,b,k,j) (a=ROR32(a,9-(j&3)),a+=b,a^=k,b=ROL32(b,1+(j&7)),b^=a)


/********************************************************************
Rotation round of XTEA cipher
********************************************************************/
#define MIXXTEA(d,s,k) ((((d << 4) ^ (d >> 5)) + d) ^ (s + k)


/********************************************************************
BOOK:    Section 6.2 "Bit mixup"
DOES:    This function mixes the bits in a 64bit value by applying
         a Speck cipher. Used by key initialization functions and
         one-time pad generation.
MOTE:    Recommended number of rounds is 27,
         USING LESS ROUNDS DECREASES RELIABILITY
RETURNS: Value pmixed[] returns the mixed bits
********************************************************************/
void Ccuser_Mix64_SPECK( 
  UNSIGNED32 pkey[Cc_KEY_LEN32], // key input 64 or 128 or 256bit
  UNSIGNED32 pdat[2],   // data input of 64 bit
  UNSIGNED32 pmixed[2], // mixed bits output of 64 bit
  UNSIGNED8 rounds      // number of mixing rounds to execute
)
{
  UNSIGNED32 d0 = pdat[0];
  UNSIGNED32 d1 = pdat[1];
  UNSIGNED32 key[Cc_KEY_LEN32];
  UNSIGNED32 tmp;
  UNSIGNED8 lp;
  
  // Get entire key
  memcpy(key,pkey,Cc_KEY_LEN8);

  MIXROUND32(d1, d0, key[0]); // apply key
  for (lp = 0; lp < rounds - 1; lp++)
  { // execute a round
    // key expansion
    MIXROUND32(key[1],key[0],lp);
#ifdef Cc_USE_ORG_SPECK
    MIXROUND32(d1,d0,key[0]); // apply key
#else
    // This is only an example of how the Speck cipher could be distorted.
    // To use: mofify the if below AND MIXROUND32x to introduce a secrete distortion.
    if ((pkey[0] & 0x33) == 0x11)
    { // CANcrypt specific distortion of Speck cipher
      // apply in 1/16th of the cases
      MIXROUND32x(d1,d0,key[0],key[1]); 
    }
    else
    { // regular Speck
      MIXROUND32(d1,d0,key[0]); // apply key
    }
#endif
    // rotate key
    tmp = key[0];
    memcpy(&(key[0]),&(key[1]),Cc_KEY_LEN8-4);
    key[Cc_KEY_LEN32-1] = tmp;
  }
  // return result 
  pmixed[0] = d0;
  pmixed[1] = d1;
}


/********************************************************************
BOOK:    Section 6.2 "Bit mixup"
DOES:    This function mixes the bits in a 64bit value by applying
         a XTEA cipher. Used by key initialization functions and
         one-time pad generation.
MOTE:    Recommended number of rounds is 64,
         USING LESS ROUNDS DECREASES RELIABILITY OF CIPHER
         Recommended key size 128bit,
         USING LESS ROUNDS DECREASES RELIABILITY OF CIPHER
RETURNS: Value pmixed[] returns the mixed bits
********************************************************************/
void Ccuser_Mix64_XTEA(
  UNSIGNED32 pkey[Cc_KEY_LEN32], // key input 64 or 128 or 256bit
  UNSIGNED32 pdat[2],   // data input of 64 bit
  UNSIGNED32 pmixed[2], // mixed bits output of 64 bit
  UNSIGNED8 rounds      // number of mixing rounds to execute
)
{
UNSIGNED32 d0 = pdat[0];
UNSIGNED32 d1 = pdat[1];
UNSIGNED32 key[Cc_KEY_LEN32];
UNSIGNED32 sum = 0;
UNSIGNED8 lp;
  
  // Get entire key
  memcpy(key,pkey,Cc_KEY_LEN8);

  for (lp=0; lp < rounds; lp++) 
  {
    d0 += MIXXTEA(d1,sum,key[sum & (Cc_KEY_LEN32-1)]));
    sum += 0x9E3779B9;
    d1 += MIXXTEA(d0,sum,key[(sum >> 11) & (Cc_KEY_LEN32-1)]));
  }
  pmixed[0] = d0;
  pmixed[1] = d1;
}


/********************************************************************
DOES:    This function mixes the bits in a 128bit value
RETURNS: Value pmixed[] returns the mixed bits
********************************************************************/
void Ccuser_Mix128( 
  UNSIGNED32 pkey[Cc_KEY_LEN32], // key input
  UNSIGNED32 pdat[4],   // data input of 128 bit
  UNSIGNED32 pmixed[4], // mixed bits output of 128 bit
  UNSIGNED8 rounds      // number of mixing rounds to execute
                        // ignored for AES
)
{
 #if (Cc_FUNCTIONALITY == Cc_SECFCT_ADVANCED)
 mbedtls_aes_context mbedtls_ctx;
  
  // Use AES
  mbedtls_aes_init(&mbedtls_ctx);
  if ( (mbedtls_aes_setkey_enc(&mbedtls_ctx,(UNSIGNED8 *)pkey,128) == 0) &&
       (mbedtls_aes_crypt_ecb(&mbedtls_ctx,MBEDTLS_AES_ENCRYPT,(UNSIGNED8 *)pdat,(UNSIGNED8 *)pmixed) == 0)
     )
  { // AES completed ok
  }
  else
  { // Error warning? Abort?
  }
 #else
  // Use Mix64
  // Execute number of rounds on both 64bit blocks
  Ccuser_Mix64(&(pkey[0]),&(pdat[0]),&(pmixed[0]),Cc_MIX64_ROUNDS);
  Ccuser_Mix64(&(pkey[0]),&(pdat[2]),&(pmixed[2]),Cc_MIX64_ROUNDS);
 #endif
}


/********************************************************************
BOOK:    Section 6.1 "Collect random numbers"
DOES:    This function expands an array with a limited number of 
         random bytes to an array of random bytes with the length
         of the current dynamic key.
RETURNS: nothing
********************************************************************/
void Ccuser_ExpandRandom(
  UNSIGNED32 pkey[Cc_KEY_LEN32],  // key input, length Cc_KEY_LEN32
  UNSIGNED32 pdest[Cc_KEY_LEN32], // destination: array with length of dyn. key
  UNSIGNED32 psrc[12]             // array with zeros and random numbers (3*15)
)
{
UNSIGNED32 pad[2];
UNSIGNED8 lp_s = 0;
UNSIGNED8 lp_d = 0;
UNSIGNED8 all_ins_used = 0;
UNSIGNED8 all_outs_used = 0;

  // init destination with permanent key
  memcpy(pdest,pkey,Cc_KEY_LEN8);
  
  while ( (all_ins_used == 0) || (all_outs_used == 0) )
  { // repeat until all input and output values have been used
    
    // ensure non-zero source
    if (psrc[lp_s] == 0)
    {
      psrc[lp_s] = pdest[lp_d];
    }
    if (psrc[lp_s+1] == 0)
    {
      psrc[lp_s+1] = pdest[lp_d+1];
    }
    
    // generate pad from permanent key and source
    // no encryption required here, just ensure values collected get mixed
    // pdest is later used as IV to generate a key
    Ccuser_Mix64(&(pkey[lp_d]),&(psrc[lp_s]),pad,(Cc_MIX64_ROUNDS >> 3)+1);

    // add pad to key
    pdest[lp_d]   += pad[0];
    pdest[lp_d+1] += pad[1];
    
    // increment offsets
    lp_d += 2;
    if (lp_d >= Cc_KEY_LEN32)
    { // end of destination reached
      lp_d = 0;
      all_outs_used = 1;
    }
    lp_s += 2;
    if (lp_s >= 12)
    { // end of source reached
      lp_s = 0;
      all_ins_used = 1;
    }
  }
}


/********************************************************************
BOOK:    Section 6.3 "Generate keys"
DOES:    Takes input from 2 keys and 1 factor to create a new key.
         Used to create a dynamic key from a permanent key using 
         random input and a serial number.
         Used to create a one-time pad from a permanent and 
         dynamic key and a counter.
RETURNS: TRUE if key initialization completed, 
         FALSE if not possible due to parameters
********************************************************************/
UNSIGNED8 Ccuser_MakeKey(
  UNSIGNED32 pin1[Cc_KEY_LEN32],// input 1: pointer to primary key used
  UNSIGNED32 pin2[Cc_KEY_LEN32],// input 2: pointer to 2nd input array
  UNSIGNED32 factor,    // input 3: optional, set zero if not used
                        // used for serial number, counter
  UNSIGNED32 pout[Cc_KEY_LEN32] // output: the dynamic key or one time pad
)
{
UNSIGNED8 ret_val = FALSE;

UNSIGNED32 fact[Cc_KEY_LEN32];
#if (Cc_KEY_LEN_BITS == 64)
UNSIGNED32 tmp_key[Cc_KEY_LEN32];
#else
UNSIGNED8 lp;
#endif

#ifdef Cc_USE_DIGOUT
  // performance measurement
  Cc_DIGOUT2_ON();
#endif

  if ( (pin1 != NULL) && (pin2 != NULL) && (pout != NULL))
  { // parameter check ok

#if (Cc_KEY_LEN_BITS == 64)
    // Here total number of rounds depending on security level
    // Speck defined value for 64bit keys: 27
    // Using less rounds greatly reduces effectiveness of security
    fact[0] = factor;
    fact[1] = ~factor;
    // Step 1: introduce factor, here mix in with pin2
    Ccuser_Mix64(pin2,fact,tmp_key,3);
    // Step 2: apply main key
    Ccuser_Mix64(pin1,tmp_key,pout,Cc_MIX64_ROUNDS);
#elif (Cc_KEY_LEN_BITS == 128)
    // Step 1: introduce factor, here mix in with pin2
    for (lp = 0; lp < Cc_KEY_LEN32; lp++)
    { // Step 1: introduce factor, here mix in with pin2
      fact[lp] = pin2[lp] ^ ROL32(factor,lp);
    }
    // Step 2: apply key
    Ccuser_Mix128(pin1,fact,pout,1); 
#else
 #error "Cc_KEY_LEN_BITS size not supported!"
#endif
  }

#ifdef Cc_USE_DIGOUT
  // performance measurement
  Cc_DIGOUT2_OFF();
#endif

  return ret_val;
}


/********************************************************************
BOOK:    Section 6.5.1 "Generate signature value"
         CANCRYPT FD ADOPTED VERSION
         Modified for CANcrypt FD, signs entire CAN FD message
DOES:    Generates a signature for a CAN FD message
RETURNS: TRUE, when generation success
********************************************************************/
UNSIGNED8 Ccuser_MakeSignature(
  CAN_MSG *pCAN,        // CAN FD message to generate signature for
  Cc_SECURITY_RECORD *pSec, // Security record, values pre-filled 
  UNSIGNED32 pKey[Cc_KEY_LEN32], // key used for signature
  UNSIGNED32 pVec[Cc_KEY_LEN32]  // key used for chksum init
)
{
UNSIGNED8 ret_val = FALSE;
UNSIGNED8 lp;
UNSIGNED8 buf_sum[Cc_KEY_LEN8];
UNSIGNED8 buf_crypt[Cc_KEY_LEN8];
UNSIGNED8 mask = Cc_KEY_LEN8-1;

#ifdef Cc_USE_DIGOUT
  Cc_DIGOUT1_ON();
#endif

  if ((pCAN != NULL) && (pSec != NULL))
  { // parameter check ok

    // initialize checksum
    if (pVec == NULL)
    {      
      memset(buf_sum,0,mask+1);
    }
    else
    {
      memcpy(buf_sum,pVec,mask+1);
    }
    
    // first add CAN ID
    buf_sum[0] += (UNSIGNED8) pCAN->id;
    buf_sum[1] += (UNSIGNED8) (pCAN->id >> 8);
    buf_sum[2] += (UNSIGNED8) (pCAN->id >> 16);
    buf_sum[3] += (UNSIGNED8) (pCAN->id >> 24) & 0x1F;
    
    // now add entire CAN Data, with security record, but without signature
    for (lp = 0; lp < (pCAN->length-2); lp++)
    { // loop through entire CAN message, without last 2 bytes
      buf_sum[(lp+4)&mask] += pCAN->dataByte[lp];
    }
#if 0
    if (pCAN->id > 0x180)
    { // only data messages
      buf_sum[0] = pCAN->length-2;
      buf_sum[1] = pSec->pad_bytes;
      buf_sum[2] = pCAN->dataByte[pCAN->length-3];
      buf_sum[3] = pCAN->dataByte[lp-1];
      DebugOutKey(0xAF,(UNSIGNED32*)buf_sum,NULL);
    }
#endif
#if 0
    if (pCAN->id > 0x180)
    { // only data messages
      pCAN->id += 0x1D000000ul + Cc_DEVICE_ID;
      pCAN->format = kCAN_FrameFormatExtend;
      CCHW_PushMessage_Internal(pCAN);
      pCAN->format = kCAN_FrameFormatStandard;
      pCAN->id -= (0x1D000000ul + Cc_DEVICE_ID);
    }
#endif

    // Now encrypt with current key
#if (Cc_KEY_LEN_BITS == 64)
    Ccuser_Mix64(pKey,(UNSIGNED32 *)buf_sum,(UNSIGNED32 *)buf_crypt,Cc_MIX64_ROUNDS); 
#elif (Cc_KEY_LEN_BITS == 128)
    Ccuser_Mix128(pKey,(UNSIGNED32 *)buf_sum,(UNSIGNED32 *)buf_crypt,1);
#else
 #error "Key size Cc_KEY_LEN_BITS not supported!"
#endif

    pSec->sign_lo = buf_crypt[0];
    pSec->sign_hi = buf_crypt[1];
    
    ret_val = TRUE;
  }
  
#ifdef Cc_USE_DIGOUT
  Cc_DIGOUT1_OFF();
#endif

  return ret_val;
}


/********************************************************************
BOOK:    Section 6.5.2 "Verify signature value"
         CANCRYPT FD ADOPTED VERSION
DOES:    Verifies a signature received in a CAN FD message
RETURNS: TRUE, if signature was verified
********************************************************************/
UNSIGNED8 Ccuser_VerifySignature(
  CAN_MSG *pCAN,        // CAN FD message to generate signature for
  Cc_SECURITY_RECORD *pSec, // Security record, values pre-filled 
  UNSIGNED32 pKey[Cc_KEY_LEN32], // key used for signature
  UNSIGNED32 pVec[Cc_KEY_LEN32]  // key used for chksum init
)
{
UNSIGNED8 ret_val = FALSE;
UNSIGNED8 chk_lo, chk_hi;

  if ((pCAN != NULL) && (pSec != NULL))
  { // parameter check ok
    
    // save org signature
    chk_lo = pSec->sign_lo;
    chk_hi = pSec->sign_hi;
    
    // now calculate signature
    if ( (Ccuser_MakeSignature(pCAN,pSec,pKey,pVec)) &&
         (chk_lo == pSec->sign_lo) && (chk_hi == pSec->sign_hi) 
       )
    { // signature match
        ret_val = TRUE;
    }
    else // mismatch
    { // restore org CAN message
      pSec->sign_lo = chk_lo;
      pSec->sign_hi = chk_hi;
    }

  }
  
  return ret_val;
}


/********************************************************************
BOOK:    Section 6.7.1 "Secure message encryption"
DOES:    Encrypts a data block in a secure message
NOTE:    This version NOT optimized for 32 bit architecture
RETURNS: TRUE if encryption completed, 
         FALSE if not possible due to parameters
********************************************************************/
UNSIGNED8 Ccuser_Encrypt(
  UNSIGNED32 ppad[Cc_KEY_LEN32], // pointer to current one-time pad
  UNSIGNED32 *pdat,     // pointer to the data to encrypt
  UNSIGNED16 first,     // first byte to encrypt
  UNSIGNED16 bytes      // number of bytes to encrypt
)
{
UNSIGNED8 ret_val = FALSE;
UNSIGNED8 lp;
UNSIGNED8 *p8pad; 
UNSIGNED8 *p8dat; 
  
  if ( (ppad != NULL) && (pdat != NULL) )
  { // parameter check ok

    p8pad = (UNSIGNED8 *) ppad; 
    p8dat = (UNSIGNED8 *) pdat; 

    for (lp = 0; lp < bytes; lp++)
    { // for length of data

      // XOR data with key, here entire pad
      p8dat[first+lp] ^= p8pad[(first+lp) & (Cc_KEY_LEN8-1)];
    }
    
    ret_val = TRUE;
  }

  return ret_val;
}


/********************************************************************
BOOK:    Section 6.7.2 "Secure message decryption"
NOTE:    Only used if cryptographic function is not symmetric and 
         decryption requires a different function then encryption
DOES:    Decrypts a data block 
RETURNS: TRUE if decryption completed, 
         FALSE if not possible due to parameters
********************************************************************/
UNSIGNED8 Ccuser_Decrypt(
  UNSIGNED32 ppad[Cc_KEY_LEN32], // pointer to current one-time pad
  UNSIGNED32 *pdat,     // pointer to the data to decrypt
  UNSIGNED16 first,     // first byte to decrypt
  UNSIGNED16 bytes      // number of bytes to decrypt
)
{
  // use encrypt function, it is symetric
  return Ccuser_Encrypt(ppad,pdat,first,bytes);
}


#ifdef __cplusplus
}
#endif
/*----------------------- END OF FILE -----------------------------*/
