/********************************************************************
MODULE:    Cc_System.h, CANcrypt System Functions Demo Module
CONTAINS:  Shared system functions of CANcrypt
COPYRIGHT: 2016-2018 Embedded Systems Academy, Inc (USA) and
           Embedded Systems Academy, GmbH (Germany)
HOME:      https://www.cancrypt.net
LICENSE:   The CANcrypt software in this distribution may be used 
           freely on NXP LPC546xx devices. When used on other devices
           or hardware, a commercial license is required, contact us
           at https://www.cancrypt.net for more information.

Unless required by applicable law or agreed to in writing, 
software distributed under the License is distributed on an 
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
either express or implied.

VERSION:   0.1, 06-AUG-2018
           $LastChangedRevision: 466 $
********************************************************************/


#include <string.h>

#ifdef __cplusplus
extern "C" {
#endif

#include "CANcrypt_includes.h"
#if (Cc_DEVICE_ID == 1)
#include "Cc_Configurator.h"
#endif


/**************************************************************************
MODULE VARIABLE
***************************************************************************/
#ifdef Cc_USE_GROUPING
// Default keys only used for key generation
static UNSIGNED32 temp_key[Cc_PERMKEY_LEN32];
#endif
  
  
/********************************************************************
Internal CANcrypt SYSTEM FUNCTIONS
********************************************************************/ 


/********************************************************************
DOES:    Retrieves a 32bit value from 4 bytes.
         The 4 bytes may be un-aligned.
RETURNS: The 32bit value
********************************************************************/
UNSIGNED32 mCc_Get32bit(
  UNSIGNED8 src[4]              // pointer to 4 bytes
  )
{
UNSIGNED8 lp = 3;
UNSIGNED32 ret_val = 0;

  while (lp > 0)
  {
    ret_val += src[lp];
    ret_val <<= 8;
    lp--;
  }
  ret_val += src[0];
  
  return ret_val;
}


/********************************************************************
DOES:    Copies a 32bit value to 4 bytes.
         The 4 bytes may be un-aligned.
RETURNS: nothing
********************************************************************/
void mCc_Put32bit(
  UNSIGNED8 dst[4],             // pointer to 4 destination bytes
  UNSIGNED32 src                // the 32bit value
)
{
UNSIGNED8 lp;
  
  for(lp=0;lp<4;lp++)
  {
    dst[lp] = (UNSIGNED8) (src);
    src >>= 8;
  }
}


/********************************************************************
DOES:    Resets all message counters. Called on pairing/grouping.
RETURNS: nothing
********************************************************************/
void mCc_ResetCounters(
  Cc_HANDLE *pCc,               // pointer to a CANcrypt handle record
  UNSIGNED32 rnd                // shared random value
)
{
  if (pCc != NULL)
  {
    pCc->err_counter = 0;
    pCc->key_up_cnt = rnd;
    pCc->grp_hb_seq = (rnd >> 16);
#ifdef Cc_USE_SECURE_MSG
    Cc_Init_Sec_Msg_Table_Counter(pCc);
#endif
  }
}


/********************************************************************
Public CANcrypt SYSTEM FUNCTIONS
********************************************************************/ 


/********************************************************************
DOES:    Resets all timeouts used.
         Cc_TIMING_FAST            0x00
         Cc_TIMING_MEDIUM          0x01
         Cc_TIMING_SLOW            0x02
         Cc_BITMETHOD_DIRECT       0x00
         Cc_BITMETHOD_DELAY        0x40
RETURNS: nothing
********************************************************************/
void Cc_SetTimeouts(
  Cc_HANDLE *pCc,               // pointer to a CANcrypt handle record
  UNSIGNED8 timing              // timing and method used
)
{
#ifdef Cc_USE_PAIRING
  pCc->key_method = timing;
  if ((timing & Cc_BITMETHOD_DELAY) == Cc_BITMETHOD_DELAY)
  { // delay mode
    if ((timing & Cc_TIMING_BITS) == Cc_TIMING_FAST)
    {
      pCc->cycle_timeout = 25;
      pCc->cycle_timernd = 0x0F;
    }
    else if ((timing & Cc_TIMING_BITS) == Cc_TIMING_MEDIUM)
    {
      pCc->cycle_timeout = 50;
      pCc->cycle_timernd = 0x1F;
    }
    else if ((timing & Cc_TIMING_BITS) == Cc_TIMING_SLOW)
    {
      pCc->cycle_timeout = 100;
      pCc->cycle_timernd = 0x3F;
    }
    else
    { // custom
      pCc->cycle_timeout = Cc_CUST_BITCYC_TIMEOUT;
      pCc->cycle_timernd = Cc_CUST_BITCYC_RANDTIME;
    }
  }
  else
  { // direct mode
    pCc->cycle_timernd = 0;
    if ((timing & Cc_TIMING_BITS) == Cc_TIMING_FAST)
    {
      pCc->cycle_timeout = 10;
    }
    else if ((timing & Cc_TIMING_BITS) == Cc_TIMING_MEDIUM)
    {
      pCc->cycle_timeout = 25;
    }
    else if ((timing & Cc_TIMING_BITS) == Cc_TIMING_SLOW)
    {
      pCc->cycle_timeout = 50;
    }
    else
    { // custom
      pCc->cycle_timeout = Cc_CUST_BITCYC_TIMEOUT;
    }
  }
#endif
}


/********************************************************************
DOES:    Adds a call-back event to queue for later processing.
         Called from interrupt processes to avoid call-back from
         interrupt level.
RETURNS: nothing
********************************************************************/
void Cc_QueueEvent(
  Cc_HANDLE  *pCc,      // pointer to CANcrypt handle record
  UNSIGNED8   event,    // event that happened (Cc_EVENT_xxx)
  UNSIGNED32  param1,   // depending on the event, up to 2 
  UNSIGNED32  param2    // event related parameters
)
{
UNSIGNED8 lp;
UNSIGNED8 overrun;
  
  overrun = TRUE;
  lp = 0;
  while ((lp < Cc_CBEVENT_QUEUE) && overrun)
  { // find first free entry in list
    if (pCc->event_irq[lp] == 0)
    { // free entry in list found
      pCc->event_irq[lp] = event;
      pCc->event_par[lp] = param1;
      pCc->event_par[Cc_CBEVENT_QUEUE+lp] = param2;
      overrun = FALSE;
    }
    lp++;
  }
  if (overrun)
  { // overrun error, change last entry to overrun
    lp = Cc_CBEVENT_QUEUE-1;
    pCc->event_irq[lp] = Cc_ERR_CBE_OVERRUN;
    pCc->event_par[lp] = 0;
    pCc->event_par[lp*2] = 0;
  }
}


/********************************************************************
PUBLIC FUNCTIONS
********************************************************************/ 

/********************************************************************
DOES:    Re-start of the CANcrypt system.
RETURNS: TRUE, if completed
         FALSE, if error in parameters passed
********************************************************************/
UNSIGNED8 Cc_Restart(
  Cc_HANDLE  *pCc,      // pointer to CANcrypt handle record
  UNSIGNED8   address,  // address of this device
  UNSIGNED32  control,  // Bit0-1: 00: No change to pairing
                        //         01: Restart pairing   
                        //         10: Stop pairing 
                        // Bit2-3: 00: No change to grouping 
                        //         01: Restart grouping
                        //         10: Stop grouping
  // Call-back functions
  Cc_CB_EVENT fct_event,      // change of status, alert
  Cc_CAN_PUSH fct_pushTxFIFO, // put CAN message into Tx FIFO
  Cc_CAN_PUSH fct_pushTxNOW,  // transmit this CAN message now
  // Device identification
  UNSIGNED32 id_1018[4] // Vendor ID, product code, revision, serial
)
{
UNSIGNED8 ret_val = FALSE;
UNSIGNED8 lp;
#ifdef Cc_USE_GROUPING
UNSIGNED8 buf[6];
#endif
  
  if ( (pCc != NULL) && (address > 0) && (address <= 15) )
  {
    CCHW_DisableCANrx();
#ifdef Cc_USE_GROUPING
    // Save grouping key selectors
    memcpy(buf,&pCc->grp_key_id_mm,6);
#endif
    // Erase entire handle
    memset(pCc,0,sizeof(Cc_HANDLE));
#ifdef Cc_USE_GROUPING
    // Restore grouping key selectors
    memcpy(&pCc->grp_key_id_mm,buf,6);
#endif
    
    memcpy(pCc->id_1018,id_1018,16);
    pCc->id_version = CANcrypt_VERSION;
    pCc->my_address = address;
    pCc->err_counter = 0;
    pCc->my_status &= ~Cc_KEY_STAT_BITS;
    pCc->my_status &= ~Cc_CMD_STAT_BITS;
    
    // Set all function pointers
    pCc->fctcb_event = fct_event;
    pCc->fct_TxFIFO = fct_pushTxFIFO;
    pCc->fct_TxNOW = fct_pushTxNOW;

    // Internal state machines
#ifdef Cc_USE_PAIRING
    pCc->my_status &= ~Cc_PAIR_STAT_BITS;
    pCc->state_key = CCkeySTATE_IDLE;
    pCc->state_pair = CCparSTATE_IDLE;
    if (control & Cc_PAIR_CTRL_STOP)
    {
      pCc->state_pair = CCparSTATE_CLOSED;
      pCc->my_status |= Cc_PAIR_STAT_FAIL;
      pCc->timer_delay = CCHW_GetTime() + 200;
    }
#endif

#ifdef Cc_USE_GROUPING
    pCc->my_status &= ~Cc_GROUP_STAT_BITS;
    if (control & Cc_GROUP_CTRL_RESTART)
    {
      pCc->state_group = CCgrpSTATE_INIT;
      // Select permanent key to use
      if ( ((pCc->grp_key_id_mm >> 5) < 2) || ((pCc->grp_key_id_mm >> 5) > 6) )
      { // Last sesssion key is default, if nothing else selected
        pCc->grp_key_id_mm = (Cc_PERM_KEY_SESSION << 5) + Cc_PERMKEY_LEN32;
      }
      // Retrieve permanent key
      pCc->pkey_perm = Ccnvol_GetPermKey(pCc->grp_key_id_mm >> 5);
      if (pCc->pkey_perm == NULL)
      { // key not available
        if ((pCc->grp_key_ctrl & Cc_GRPKEY_GENERATE) != 0)
        { // start key generation cycle
          // generate a default key for grouping
          temp_key[0] = 0xDEADC0DE + pCc->grp_key_id_mm;
          for (lp = 1; lp < Cc_PERMKEY_LEN32; lp++)
          { // generate some default initialization key
            temp_key[lp] = (temp_key[lp-1] + 0xBADC0FFE) ^ (((UNSIGNED32)pCc->grp_key_id_mm << 18) + lp);
          }
          Ccuser_MakeKey(temp_key,&(temp_key[Cc_KEY_LEN32]),pCc->grp_key_id_mm,&(temp_key[Cc_KEY_LEN32/2]));
          // use this as initialization key
          pCc->pkey_perm = temp_key;
          // this is now a key generation cycle, inform application
          pCc->fctcb_event(Cc_ERR_GROUP_KEYGEN,pCc->grp_key_id_mm >> 5,0x80 + (pCc->grp_key_id_mm & 0x7F));
        }
        else
        { // try to find next higher priority key available
          while ( (pCc->pkey_perm == NULL) && ((pCc->grp_key_id_mm >> 5) < Cc_PERM_KEY_MANUFACTURER) )
          { // find next key available
            pCc->grp_key_id_mm += (1 << 5);
            pCc->pkey_perm = Ccnvol_GetPermKey(pCc->grp_key_id_mm >> 5);
          }
          if (pCc->pkey_perm == NULL)
          { // still no key found
            pCc->state_group = CCgrpSTATE_CLOSED;
            pCc->my_status |= Cc_GROUP_STAT_FAIL;
            pCc->fctcb_event(Cc_ERR_KEY_NO_KEY,Cc_PERM_KEY_SESSION,0);
          }
        }
      }
      else if ((pCc->grp_key_ctrl & Cc_GRPKEY_GENERATE) != 0)
      { // key is available, do not allow overwrite
        pCc->fctcb_event(Cc_ERR_GROUP_KEYNOGEN,pCc->grp_key_id_mm >> 5,0x80 + (pCc->grp_key_id_mm & 0x7F));
        pCc->grp_key_ctrl &= 0x07;
      }
    }
    else if (control & Cc_GROUP_CTRL_STOP)
    {
      pCc->state_group = CCgrpSTATE_CLOSED;
      pCc->my_status |= Cc_GROUP_STAT_FAIL;
    }
#endif 
    
#ifdef Cc_USE_MONITORING
    Cc_Init_IDList(&(pCc->unx_lst));
#endif

#ifdef Cc_USE_SECURE_MSG
    pCc->pMsgTblRx = NULL;
    pCc->pMsgTblTx = NULL;
    pCc->pMsgCntRx = NULL;
    pCc->pMsgCntTx = NULL;
#endif

    // CAN messages for configurator and device, init ID and format only
    pCc->can_config.id = Cc_CANID_CONFIG;
    pCc->can_config.format = kCAN_FrameFormatStandard;
    pCc->can_config.type = kCAN_FrameTypeData;
    pCc->can_config.proto = kCAN_ProtoTypeFD;
    pCc->can_config.bitratemode = kCAN_BitrateModeTypeSwitch;
    pCc->can_device.id = Cc_CANID_CONFIG+pCc->my_address-1;
    pCc->can_device.format = kCAN_FrameFormatStandard;
    pCc->can_device.type = kCAN_FrameTypeData;
    pCc->can_device.proto = kCAN_ProtoTypeFD;
    pCc->can_device.bitratemode = kCAN_BitrateModeTypeSwitch;
    // all done, allow interrupt again
    CCHW_EnableCANrx();
    ret_val = 1;
  }
  return ret_val;
}


#ifdef Cc_USE_GROUPING
/********************************************************************
DOES:    Selects the next grouping event
         If none specified, last saved session key and group is used.
RETURNS: TRUE, if key availanle, else FALSE
********************************************************************/
UNSIGNED8 Cc_SelectGroup(
  Cc_HANDLE  *pCc,      // pointer to CANcrypt handle record
  UNSIGNED8  key_major, // key ID major info (2-6)
  UNSIGNED8  key_minor, // key ID minor info (size in 32bits)
  UNSIGNED16 grp_exp ,  // Expected devices (bit 0 = unused, 
                        // bit 1 = Device1, bit 2 = Device 2, etc.)
  UNSIGNED16 grp_opt    // Optional devices
)
{
  UNSIGNED32 *pKey;
  UNSIGNED8 ret_val = FALSE;
  
  // first check, if key is available
  pKey = Ccnvol_GetPermKey(key_major);
  if (pKey == NULL)
  { // key not available
    if ((key_major & Cc_GRPKEY_GENERATE) != 0)
    { // requesting key generation cycle for this
      ret_val = TRUE;
      pCc->grp_key_ctrl = key_major;
    }
  }
  else
  { // key is available
    if ((key_major & Cc_GRPKEY_GENERATE) == 0)
    { // no key generation request, just use
      ret_val = TRUE;
    }
  }
  if (ret_val)
  {
    pCc->grp_key_id_mm = (key_major << 5) + key_minor;
    pCc->grp_info_exp = grp_exp;
    pCc->grp_info_opt = grp_opt;
  }
  
  return ret_val;
}
#endif


#ifdef Cc_USE_PAIRING
/********************************************************************
DOES:    Re-start of a generic access communication channel.
RETURNS: TRUE, if completed
         FALSE, if error in parameters passed
********************************************************************/
UNSIGNED8 Cc_Restart_GenAccess(
  Cc_GEN_ACC_HANDLE  *pGA,    // pointer to generall access handle
  Cc_GA_READ  fct_readacc,    // generic read access
  Cc_GA_WRITE fct_writeacc,   // generic write access
  UNSIGNED8   partner         // communication partner (1-15)
)
{
UNSIGNED8 ret_val = FALSE;
  
  if (pGA != NULL)
  {
    pGA->fct_GARead = fct_readacc;
    pGA->fct_GAWrite = fct_writeacc;
    // generic access variables
    pGA->partner = partner;
    pGA->gencnt = 0;
  }

  return ret_val;
}
#endif


/********************************************************************
DOES:    Disconnect from the CANcrypt communication parners,
         sends a request to end pairing / grouping. 
RETURNS: nothing
********************************************************************/
void Cc_TxDisconnect(
  Cc_HANDLE *pCc,       // pointer to CANcrypt handle record
  UNSIGNED8 dest_addr,  // paired device ID (1-15) or 0 for group
  UNSIGNED8 reason      // reason for disconnecting, event/alert code
)
{
Cc_SECURITY_RECORD *pSec; // CANcryprt security record
UNSIGNED32 rnd;
CAN_MSG can;
  
  if ((pCc->my_status & Cc_GROUP_STAT_BITS) == Cc_GROUP_STAT_GROUPED)
  {
#ifdef CANcrypt_DEBUG_DETAIL
    DebugOutKey(0xD1,NULL,pCc);
#endif
    can.id = Cc_CANID_CONFIG+pCc->my_address-1;
    can.dataByte[0] = (dest_addr << 4) + Cc_CTRL_UNPAIRGROUP;
    can.dataByte[1] = pCc->my_status;
    can.dataByte[2] = reason;
    can.dataByte[4] = 0;
    can.length = 12;
    
    // now add security record
    pSec = (Cc_SECURITY_RECORD *) &(can.dataByte[4]);
    rnd = CCHW_Rand();
    pSec->rnd_lo = (UNSIGNED8) rnd;
    pSec->rnd_hi = (UNSIGNED8) (rnd >> 8);
    pSec->seq_lo = (UNSIGNED8) pCc->grp_hb_seq;
    pSec->seq_hi = (UNSIGNED8) (pCc->grp_hb_seq >> 8);
    pSec->pad_bytes = 0;
    pSec->encrypt = 0;
    pSec->active = 1;
    pSec->key_id = Cc_DYN_KEY_GROUP;
    pSec->key_cnt = pCc->key_up_cnt;
    // Generate signature
    if (Ccuser_MakeSignature(&can,pSec,pCc->key_dyn,&(pCc->pkey_perm[Cc_PERMKEY_LEN32-Cc_KEY_LEN32])))
    {
      can.format = kCAN_FrameFormatStandard;
      can.type = kCAN_FrameTypeData;
      can.proto = kCAN_ProtoTypeFD;
      can.bitratemode = kCAN_BitrateModeTypeSwitch;
      pCc->fct_TxFIFO(&can);
    }
  }
  
#ifdef Cc_USE_GROUPING
  // ungroup after next sec HB
  pCc->grp_info_use = 0;
  pCc->grp_info_opt = reason;
#endif // Cc_USE_GROUPING
}


/*****************************************************************************
DOES:    Generates an acknowledge or abort
RETURNS: nothing
*****************************************************************************/
void Cc_TxAckAbort(
  Cc_HANDLE *pCc,               // pointer to a CANcrypt handle record
  UNSIGNED8 ack,                // TRUE for Ack, FALSE for Abort
  UNSIGNED8 dest_addr,          // destination device ID (1-15) 
                                // or 0 for broadcast
  UNSIGNED8 key_id,             // key id for this acknowledge, 0 if unknown
  UNSIGNED8 key_len             // key len for this acknowledge, 0 if unknown
)
{
Cc_SECURITY_RECORD *pSec; // CANcryprt security record
UNSIGNED32 rnd;
CAN_MSG can;
  
  can.id = Cc_CANID_CONFIG+pCc->my_address-1;
  if (ack)
  {
    can.dataByte[0] = (dest_addr << 4) + Cc_CTRL_ACK;
  }
  else
  {
    can.dataByte[0] = (dest_addr << 4) + Cc_CTRL_ABORT;
  }
  can.dataByte[1] = pCc->my_status;
  can.dataByte[2] = key_id;
  can.dataByte[3] = key_len;
  can.length = 12;

  if ((pCc->my_status & Cc_GROUP_STAT_BITS) == Cc_GROUP_STAT_GROUPED)
  { // only add signature if grouped or paired
    rnd = CCHW_Rand();
#ifdef CANcrypt_DEBUG_DETAIL
   DebugOutKey(0xAA,NULL,pCc);
   // DebugOutCAN(Cc_DEVICE_ID,4,pCc->pkey_perm[0],pCc->pkey_perm[0]>>8,pCc->pkey_perm[0]>>16,pCc->pkey_perm[0]>>24,0,0,0,0);
#endif
  
    // now add security record
    pSec = (Cc_SECURITY_RECORD *) &(can.dataByte[4]);
    rnd = CCHW_Rand();
    pSec->rnd_lo = (UNSIGNED8) rnd;
    pSec->rnd_hi = (UNSIGNED8) (rnd >> 8);
    pSec->seq_lo = (UNSIGNED8) pCc->grp_hb_seq;
    pSec->seq_hi = (UNSIGNED8) (pCc->grp_hb_seq >> 8);
    pSec->pad_bytes = 0;
    pSec->encrypt = 0;
    pSec->active = 1;
    pSec->key_id = Cc_DYN_KEY_GROUP;
    pSec->key_cnt = pCc->key_up_cnt;
    // Generate signature
    if (!Ccuser_MakeSignature(&can,pSec,pCc->key_dyn,&(pCc->pkey_perm[Cc_PERMKEY_LEN32-Cc_KEY_LEN32])))
    {
      pSec->sign_lo = 0xFF;
      pSec->sign_hi = 0xFF;
    }
  }
  
  // send out message
  can.format = kCAN_FrameFormatStandard;
  can.type = kCAN_FrameTypeData;
  can.proto = kCAN_ProtoTypeFD;
  can.bitratemode = kCAN_BitrateModeTypeSwitch;
  pCc->fct_TxFIFO(&can);

}


/*****************************************************************************
DOES:    Generates an alert message
RETURNS: nothing
*****************************************************************************/
void Cc_TxAlert(
  Cc_HANDLE *pCc,               // pointer to a CANcrypt handle record
  UNSIGNED8 dest_addr,          // destination device ID (1-15) 
                                // or 0 for broadcast
  UNSIGNED16 alert              // 16bit alert / error code
)
{
Cc_SECURITY_RECORD *pSec; // CANcryprt security record
UNSIGNED32 rnd;
CAN_MSG can;
  
  can.id = Cc_CANID_CONFIG+pCc->my_address-1;
  can.dataByte[0] = (dest_addr << 4) + Cc_CTRL_ALERT;
  can.dataByte[1] = pCc->my_status;
  can.dataByte[2] = (UNSIGNED8) alert;
  can.dataByte[3] = alert >> 8;
  can.length = 12;
  
  if ((pCc->my_status & Cc_GROUP_STAT_BITS) == Cc_GROUP_STAT_GROUPED)
  { // only sign when grouped
    can.id = Cc_CANID_CONFIG+pCc->my_address-1;
    can.dataByte[0] = (dest_addr << 4) + Cc_CTRL_ALERT;
    can.dataByte[1] = pCc->my_status;
    can.dataByte[2] = (UNSIGNED8) alert;
    can.dataByte[3] = alert >> 8;
    can.length = 12;
    
    // now add security record
    pSec = (Cc_SECURITY_RECORD *) &(can.dataByte[4]);
    rnd = CCHW_Rand();
    pSec->rnd_lo = (UNSIGNED8) rnd;
    pSec->rnd_hi = (UNSIGNED8) (rnd >> 8);
    pSec->seq_lo = (UNSIGNED8) pCc->grp_hb_seq;
    pSec->seq_hi = (UNSIGNED8) (pCc->grp_hb_seq >> 8);
    pSec->pad_bytes = 0;
    pSec->encrypt = 0;
    pSec->active = 1;
    pSec->key_id = Cc_DYN_KEY_GROUP;
    pSec->key_cnt = pCc->key_up_cnt;
    // Generate signature
    if (!Ccuser_MakeSignature(&can,pSec,pCc->key_dyn,&(pCc->pkey_perm[Cc_PERMKEY_LEN32-Cc_KEY_LEN32])))
    {
      pSec->sign_lo = 0xFF;
      pSec->sign_hi = 0xFF;
    }
  }
  can.format = kCAN_FrameFormatStandard;
  can.type = kCAN_FrameTypeData;
  can.proto = kCAN_ProtoTypeFD;
  can.bitratemode = kCAN_BitrateModeTypeSwitch;
  pCc->fct_TxFIFO(&can);
}


/*****************************************************************************
DOES:    Generates an identify message
RETURNS: nothing
*****************************************************************************/
void Cc_TxIdentify(
  Cc_HANDLE *pCc,               // pointer to a CANcrypt handle record
  UNSIGNED16 version,           // CANcrypt version
  UNSIGNED8 key_id,             // key id desired
  UNSIGNED8 key_len,            // key len desired
  UNSIGNED8 cc_method,          // method desired (Cc_SECFCT_xxx)
  UNSIGNED8 cc_timing           // timing desired (Cc_TIMING_xxx)
)
{
  pCc->can_ident.id = Cc_CANID_CONFIG+pCc->my_address-1;
  pCc->can_ident.dataByte[0] = Cc_CTRL_IDENTIFY;
  pCc->can_ident.dataByte[1] = pCc->my_status;
  pCc->can_ident.dataByte[2] = key_id;
  pCc->can_ident.dataByte[3] = key_len;
  pCc->can_ident.dataByte[4] = (UNSIGNED8) version;
  pCc->can_ident.dataByte[5] = version >> 8;
  pCc->can_ident.dataByte[6] = (cc_method << 4) + cc_timing;
  pCc->can_ident.length = 7;
  pCc->key_method = (cc_method << 4) | cc_timing;
}


/********************************************************************
DOES:    Main CANcrypt householding functionality. Call cyclicly.
         Primarily monitors timeouts and state transitions.
RETURNS: TRUE, if there was something to process
         FALSE, if there was nothing to do
********************************************************************/
UNSIGNED8 Cc_Process_Tick(
  Cc_HANDLE *pCc        // pointer to a handle record
)
{
UNSIGNED16 cmp;
UNSIGNED8 ret_val = FALSE;

  #if (Cc_DEVICE_ID != 1)
CAN_MSG cantx;
  
  cantx.format = kCAN_FrameFormatStandard;
  cantx.type = kCAN_FrameTypeData;
  cantx.proto = kCAN_ProtoTypeFD;
  cantx.bitratemode = kCAN_BitrateModeTypeSwitch;
#endif
  
  // check if handle is initialized
  if ( (pCc != NULL) && (pCc->id_version == CANcrypt_VERSION) &&
       ((pCc->id_1018[0] != 0ul) && (pCc->id_1018[0] != 0xFFFFFFFFul))
     )
  { // Handle properly initialized
  
    // first check if there are call back events
    cmp = 0;
    while ((pCc->event_irq[cmp] != 0) && (cmp < Cc_CBEVENT_QUEUE))
    {
      CCHW_DisableCANrx();
      pCc->fctcb_event(pCc->event_irq[cmp],pCc->event_par[cmp],pCc->event_par[Cc_CBEVENT_QUEUE+cmp]);
      if (pCc->event_irq[cmp] >= Cc_EVENT_GENERR) 
      {
        Cc_TxAlert(pCc,0,pCc->event_irq[cmp]);
      }
      pCc->event_irq[cmp] = 0;
      CCHW_EnableCANrx();
      cmp++;
    }
    
    // is there an (extended)identification message to transmit?
    if (pCc->can_ident.id != 0)
    {
      CCHW_DisableCANrx();
      pCc->fct_TxFIFO(&pCc->can_ident);
      // mark as handled
      pCc->can_ident.id = 0;
      CCHW_EnableCANrx();
    }
    
    // now check error counter
    if (pCc->err_counter >= 128)
    { // error counter reached critical level, unpair
#ifdef Cc_USE_PAIRING
      pCc->my_status &= ~Cc_PAIR_STAT_BITS;
      pCc->state_key = CCkeySTATE_CLOSED;
      pCc->state_pair = CCparSTATE_CLOSED; 
#endif
#ifdef Cc_USE_GROUPING
      pCc->my_status &= ~Cc_GROUP_STAT_BITS;
      pCc->state_group = CCgrpSTATE_CLOSED;
#endif
      pCc->timer_delay = CCHW_GetTime() + 200;
      // do not come back here
      pCc->err_counter = 127;
      pCc->fctcb_event(Cc_ERR_ERRCOUNT,0,0);
    }

  #ifdef Cc_USE_MONITORING
    // did we receive an unexpected message?
    ret_val = Cc_Process_Monitor_Tick(pCc);
  #endif
    
  #ifdef Cc_USE_SECURE_MSG
    if (! ret_val)
    { // secure message handling OK?
      ret_val = Cc_Process_secMsg_Tick(pCc);
    }
  #endif  

  #ifdef Cc_USE_PAIRING
    if (! ret_val)
    { // work on key generation timings and timeouts
      ret_val = Cc_Process_Key_Tick(pCc);
      if (! ret_val)
      { // work on pairing timings and timeouts
        ret_val = Cc_Process_Pair_Tick(pCc);
      }
    }
  #endif
    
  #ifdef Cc_USE_GROUPING
    if (! ret_val)
    { // work on grouping timings and timeouts
      ret_val = Cc_Process_Group_Tick(pCc);
    }
  #endif
    
#if (Cc_DEVICE_ID != 1)
    if (pCc->identify[3] == 1)
    { // identify request received, send response
      
      cantx.dataByte[0] = 0x10 + Cc_CTRL_IDENTIFY; // reply to configurator
      cantx.dataByte[1] = pCc->my_status;
      if ((pCc->identify[0] != 0xFF) || (pCc->identify[1] != 0xFF))
      { // requested key available
        cantx.dataByte[2] = pCc->identify[0];
        cantx.dataByte[3] = pCc->identify[1];
      }
      else
      { // request was wrong, answer with what is available
        cantx.dataByte[2] = Cc_PERMKEY_DEFAULT;
        cantx.dataByte[3] = 0x80 + Cc_PERMKEY_LEN32;
      }
      cantx.dataByte[4] = (UNSIGNED8) CANcrypt_VERSION;
      cantx.dataByte[5] = (UNSIGNED8) (CANcrypt_VERSION >> 8);
      if (pCc->key_method == 0)
      { // If timeouts not yet set, apply defaults
        Cc_SetTimeouts(pCc,Cc_METHOD);
      }
      cantx.dataByte[6] = pCc->key_method;
      cantx.id = Cc_CANID_CONFIG + pCc->my_address - 1;
      cantx.length = 7;
      pCc->fct_TxFIFO(&(cantx));

      // erase request
      pCc->identify[3] = 0;
    }
#endif
#if 0
    else if (pCc->identify[3] == 2)
    { // extended identify request received, send response
      cantx.dataByte[0] = 0x10 + Cc_CTRL_XTDID; // reply to configurator
      cantx.dataByte[1] = pCc->my_status;
      idx = pCc->identify[1];
      idx <<= 8;
      idx += pCc->identify[0];
      mCc_Put32bit(&(cantx.dataByte[3]),0ul); // init data with zero
      // !!! ???
      //cantx.dataByte[2] = mCc_GenRead(pCc,FALSE,idx,pCc->identify[2],&(cantx.dataByte[3]));
      cantx.id = Cc_CANID_CONFIG + pCc->my_address - 1;
      cantx.length = 7;
      pCc->fct_TxFIFO(&(cantx));

      // erase request
      pCc->identify[3] = 0;
    }

    if ( (pCc->genacc.can1.id >= Cc_CANID_CONFIG) && 
         (pCc->genacc.can1.id < Cc_CANID_CONFIG+15) &&
         (pCc->genacc.can1.id != (Cc_CANID_CONFIG+pCc->my_address-1)) &&
         ( ((pCc->genacc.can1.dataByte[0] & 0x0F) == Cc_CTRL_DATRD) ||
           ((pCc->genacc.can1.dataByte[0] & 0x0F) == Cc_CTRL_DATWR)
         )
       )
    { // CANcrypt generic read/write request received
      if ( (pCc->genacc.partner != 0) &&
           (pCc->genacc.can1.id == pCc->genacc.can2.id) && 
           (pCc->genacc.can1.dataByte[0] == pCc->genacc.can2.dataByte[0])
         )
      { // both parts of request present, call handler
        if (mCc_RxGenAccess(pCc))
        { // request authorized
          idx = pCc->genacc.can1.dataByte[3];
          idx <<= 8;
          idx += pCc->genacc.can1.dataByte[2];
          if (idx == 0xFFFF)
          { // response to this device
#if (Cc_DEVICE_ID == 1)
            // if this is configurator, automatically trigger key update
            // set marker that we came from here
            pCc->genacc.can1.id = 0xACCE;
            Ccconfig_Request_Update_Key(pCc);
#else
            // mark requests as handled
            pCc->genacc.can1.id = 0;
            pCc->genacc.can2.id = 0;
            // call callback function to indicate end of read/write
            if ((pCc->genacc.can1.dataByte[0] & 0x0F) == Cc_CTRL_DATRD)
            { // read response
              pCc->fctcb_event(Cc_GACC_READ_OK,pCc->genacc.can2.dataByte[2]&0x07,mCc_Get32bit(&(pCc->genacc.can2.dataByte[4])));
            }
            else
            { // write response
              pCc->fctcb_event(Cc_GACC_WRITE_OK,0,0);
            }
#endif
          }
          else
          { // request to this device
            if ((pCc->genacc.can1.dataByte[0] & 0x0F) == Cc_CTRL_DATRD)
            { // read
              cmp = mCc_GenRead(pCc,TRUE,idx,pCc->genacc.can1.dataByte[4],&(pCc->genacc.can2.dataByte[4]));
              // set length
              pCc->genacc.can2.dataByte[2] = (UNSIGNED8) cmp;
              rnd = CCHW_Rand();
              pCc->genacc.can2.dataByte[3] = rnd;
              // fill unsued data with random data
              rnd = CCHW_Rand();
              while (cmp < 4)
              {
                pCc->genacc.can2.dataByte[4+cmp] = (UNSIGNED8) (rnd >> (cmp * 8));
                cmp++;
              }
            }
            else
            { // write
              pCc->genacc.can2.dataByte[2] =  mCc_GenWrite(pCc,idx,pCc->genacc.can1.dataByte[4],pCc->genacc.can2.dataByte[2]&0x07,&(pCc->genacc.can2.dataByte[4]));
              rnd = CCHW_Rand();
              mCc_Put32bit(&(pCc->genacc.can2.dataByte[3]),rnd);
              rnd = CCHW_Rand();
              pCc->genacc.can2.dataByte[7] = rnd;
            }
            // prepare response
            pCc->genacc.can1.id = Cc_CANID_CONFIG + pCc->my_address - 1;
            pCc->genacc.can2.id = Cc_CANID_CONFIG + pCc->my_address - 1;
            pCc->genacc.can1.dataByte[0] &= 0x0F;
            pCc->genacc.can1.dataByte[0] |= (pCc->genacc.partner << 4);
            pCc->genacc.can2.dataByte[0] &= 0x0F;
            pCc->genacc.can2.dataByte[0] |= (pCc->genacc.partner << 4);
            pCc->genacc.can1.dataByte[1] = pCc->my_status;
            pCc->genacc.can2.dataByte[1] = pCc->my_status;
            pCc->genacc.can1.dataByte[2] = 0xFF; // mark as response
            pCc->genacc.can1.dataByte[3] = 0xFF;
            pCc->genacc.can1.dataByte[5] = pCc->genacc.gencnt;
            // send response
            mCc_TxGenAccess(pCc,&pCc->genacc.can1,&pCc->genacc.can2,&pCc->genacc.gencnt,3,5);
            // mark requests as handled
            pCc->genacc.can1.id = 0;
            pCc->genacc.can2.id = 0;
          }
        }
        else
        { // request not authorized
          // mark requests as handled
          pCc->genacc.can1.id = 0;
          pCc->genacc.can2.id = 0;
          // send abort message
          Cc_TxAckAbort(pCc,FALSE,pCc->genacc.partner,0,0);
          // callback - abort
          pCc->fctcb_event(Cc_ERR_ABORT, pCc->my_status, 0);
        }
      }
    }
#endif

  }

  return ret_val;
}


/********************************************************************
DOES:    This is the main CAN receive function of CANcrypt, must be
         called directly from CAN receive interrupt. Distributes a
         message to the other Cc_Process_xxx_Rx() functions.
RETURNS: TRUE, if this message was processed by CANcrypt
         FALSE, if this message was ignored by CANcrypt
********************************************************************/
UNSIGNED8 Cc_Process_Rx(
  Cc_HANDLE *pCc,       // pointer to CANcrypt handle record
  CAN_MSG *pCANrx       // pointer to CAN message received
)
{
UNSIGNED8 ret_val = FALSE;
#ifdef _MSC_VER
CC_TIMER_TYPE delay_timestamp;
#endif
  
  // check if handle is initialized
  if ( (pCANrx != NULL) && 
       (pCc != NULL) && (pCc->id_version == CANcrypt_VERSION) &&
       ((pCc->id_1018[0] != 0ul) && (pCc->id_1018[0] != 0xFFFFFFFFul))
     )
  { // Handle properly initialized

  #ifdef Cc_USE_MONITORING
    // is message in list of unexpected messages?
    ret_val = Cc_Process_Monitor_Rx(pCc,pCANrx);
  #endif
    
  #ifdef Cc_USE_PAIRING
    if (! ret_val)
    { // is message for pairing?
      ret_val =  Cc_Process_Pair_Rx(pCc,pCANrx);
      if (! ret_val)
      { // is message for key generation?
        ret_val = Cc_Process_Key_Rx(pCc,pCANrx);
      }
    }
  #endif

  #ifdef Cc_USE_GROUPING
    if (! ret_val)
    { // is message for grouping?
      ret_val =  Cc_Process_Group_Rx(pCc,pCANrx);
    }
  #endif
    
  #ifdef Cc_USE_SECURE_MSG
    if (! ret_val)
    { // is message a secure message?
      ret_val = Cc_Process_secMsg_Rx(pCc,pCANrx);
      if ((ret_val & 1) == 1)
      { // message was processed, do not add to FIFO
        ret_val = TRUE;
      }
      else
      {
        ret_val = FALSE;
      }
    }
  #endif
    
    if (! ret_val)
    { // is this a kernel request?

#ifdef _MSC_VER
      // for certain messages add a delay to allow time for CANcrypt to be ready for the message
      if (((pCANrx->id >= Cc_CANID_DEBUG)  && (pCANrx->id < Cc_CANID_DEBUG+15))  ||
          ((pCANrx->id >= Cc_CANID_CONFIG) && (pCANrx->id < Cc_CANID_CONFIG+15))
         )
      {
        delay_timestamp = CCHW_GetTime() + 3;
        while (!CCHW_IsTimeExpired(delay_timestamp));
      }
#endif

      if ( (pCANrx->id >= Cc_CANID_CONFIG) && (pCANrx->id < Cc_CANID_CONFIG+15) &&
           (((pCANrx->dataByte[0] & 0xF0) == 0) || ((pCANrx->dataByte[0] >> 4) == pCc->my_address))
          )
      { // request directed at this device 
        if ((pCANrx->dataByte[0] >> 4) == pCc->my_address)
        { // directed directly at us (not broadcast)
          if ((pCANrx->dataByte[0] & 0x0F) == Cc_CTRL_ABORT)
          { // abort message directed to this node
#ifdef Cc_USE_PAIRING
            pCc->my_status |= Cc_PAIR_STAT_FAIL;
            pCc->state_key = CckeySTATE_ABORT;
            pCc->state_pair = CCparSTATE_CLOSED;
#endif
#ifdef Cc_USE_GROUPING
            pCc->state_group = CCgrpSTATE_CLOSED;
#endif
            pCc->timer_delay = CCHW_GetTime() + 200;
            Cc_QueueEvent(pCc,Cc_ERR_ABORT_PAR,pCANrx->id-Cc_CANID_CONFIG+1,0);
          }
        }
#if (Cc_DEVICE_ID != 1)
        else
        { // broadcast request
          if ((pCANrx->dataByte[0] & 0x0F) == Cc_CTRL_IDENTIFY)
          { // identify request
            if ( (Ccnvol_GetPermKeyID(pCANrx->dataByte[2]) != 0ul) && 
                 (pCANrx->dataByte[3] == (0x80 + Cc_PERMKEY_LEN32)) &&
                 (pCANrx->dataByte[4] == (UNSIGNED8) CANcrypt_VERSION) &&
                 (pCANrx->dataByte[5] == (UNSIGNED8) (CANcrypt_VERSION >> 8)) &&
                 (pCANrx->length == 7) &&
                 (((pCANrx->dataByte[6]>>4) & Cc_BITMETHOD_DELAY) == Cc_BITMETHOD) &&
                 (((pCANrx->dataByte[6]>>4) & Cc_BITMETHOD_16CLAIMS) == Cc_BITMETHOD_CLAIMS) &&
                 ((pCANrx->dataByte[6] & Cc_SECFCT_CUSTOM) == Cc_FUNCTIONALITY)
               )
            { // it is an identify request, save key info
              pCc->identify[0] = pCANrx->dataByte[2];
              pCc->identify[1] = pCANrx->dataByte[3];
              pCc->identify[2] = pCANrx->dataByte[4]; // method
              pCc->identify[3] = 1; // regular request
            }
            else
            { // requested key not available
              pCc->identify[0] = 0xFF;
              pCc->identify[1] = 0xFF;
              pCc->identify[3] = 1; // regular request
              Cc_QueueEvent(pCc,Cc_ERR_IDENT,CANcrypt_VERSION,pCANrx->dataByte[6]);
            }
          }
        }
#endif
        ret_val = TRUE;
      }
    }
    
  }
    
  return ret_val;
}


#ifdef __cplusplus
}
#endif
/*----------------------- END OF FILE --------------------------------------*/
