/********************************************************************
MODULE:    Cc_Grouping.c, CANcrypt Grouping Demo Module
CONTAINS:  CANcrypt grouping functionality
COPYRIGHT: 2016-2018 Embedded Systems Academy, Inc (USA) and
           Embedded Systems Academy, GmbH (Germany)
HOME:      https://www.cancrypt.net
LICENSE:   The CANcrypt software in this distribution may be used 
           freely on NXP LPC546xx devices. When used on other devices
           or hardware, a commercial license is required, contact us
           at https://www.cancrypt.net for more information.

Unless required by applicable law or agreed to in writing, 
software distributed under the License is distributed on an 
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
either express or implied.

VERSION:   0.1, 06-AUG-2018
           $LastChangedRevision: 466 $
********************************************************************/

#include <string.h>

#ifdef __cplusplus
extern "C" {
#endif

#include "CANcrypt_includes.h"
#include "board.h"

#ifdef Cc_USE_GROUPING

#ifdef CANcrypt_DEBUG_DETAIL
 #ifndef _MSC_VER
 #warning "DEBUG ACTIVE: sharing pads and keys"
 #endif
#endif

/**************************************************************************
LOCAL DEFINITIONS
***************************************************************************/ 


/**************************************************************************
MODULE VARIABLE
***************************************************************************/ 


/**************************************************************************
LOCAL FUNCTIONS
***************************************************************************/ 

// returns number of bits set
UNSIGNED8 NrOfBitsSetIn (UNSIGNED16 dat)
{
  UNSIGNED8 count = 0;
  while(dat)
  {
    count += dat & 1;
    dat >>= 1;
  }
  return count;
}


/**************************************************************************
PUBLIC FUNCTIONS
***************************************************************************/ 

/**************************************************************************
DOES:    Main CANcrypt grouping householding functionality. Call cyclicly.
RETURNS: TRUE, if there was something to process
         FALSE, if there was nothing to do
**************************************************************************/
UNSIGNED8 Cc_Process_Group_Tick(
  Cc_HANDLE *pCc        // Pointer to handle with CANcrypt data
)
{
UNSIGNED32 rand_key[Cc_KEY_LEN32];
UNSIGNED32 rnd; // random value
Cc_SECURITY_RECORD *pSec; // CANcryprt security record
UNSIGNED8 verify;
UNSIGNED8 ret_val = FALSE;

  switch(pCc->state_group)
  {
  case CCgrpSTATE_INIT:
    // are all parameters in place for grouping?
    if ( ((pCc->grp_info_exp & 1) == 0) && (pCc->pkey_perm != 0) && 
         (pCc->my_address > 0) && (pCc->my_address <= 15)
       )
    { // we have grouping info, permanent key and and address
      // update own status
      pCc->my_status &= ~Cc_GROUP_STAT_BITS;
      pCc->my_status |= Cc_GROUP_STAT_PROGRESS;
      // re-init variables used
      memset(&pCc->key_rnd,0,48);
      pCc->grp_info_fnd = (1 << pCc->my_address);
      pCc->grp_HBevent = Cc_SECHB_EVENT_TIME;
      pCc->grp_HBinhib = Cc_SECHB_INHIBIT_TIME;
      // prepare initial transmit
      pCc->can_device.dataByte[0] = Cc_CTRL_PAIRGROUP;
      pCc->can_device.dataByte[1] = pCc->my_status;
      pCc->can_device.dataByte[2] = 2; // key ID;
      pCc->can_device.dataByte[3] = 0x80 + Cc_KEY_LEN32;
      rnd = CCHW_Rand();
      pCc->can_device.dataByte[4] = rnd;
      pCc->can_device.dataByte[5] = rnd >> 8;
      pCc->can_device.dataByte[6] = rnd >> 16;
      pCc->can_device.dataByte[7] = NrOfBitsSetIn(pCc->grp_info_exp);
      // add random values to string of collected values
      pCc->key_rnd[ (pCc->my_address-1)*3   ] = pCc->can_device.dataByte[4];
      pCc->key_rnd[((pCc->my_address-1)*3)+1] = pCc->can_device.dataByte[5];
      pCc->key_rnd[((pCc->my_address-1)*3)+2] = pCc->can_device.dataByte[6];
      // now add security record
      pSec = (Cc_SECURITY_RECORD *) &(pCc->can_device.dataByte[8]);
      rnd = CCHW_Rand();
      pSec->rnd_lo = (UNSIGNED8) rnd;
      pSec->rnd_hi = (UNSIGNED8) (rnd >> 8);
      pSec->seq_lo = (UNSIGNED8) pCc->grp_hb_seq;
      pSec->seq_hi = (UNSIGNED8) (pCc->grp_hb_seq >> 8);
      pSec->pad_bytes = 0;
      pSec->encrypt = 0;
      pSec->active = 1;
      pSec->key_id = (pCc->grp_key_id_mm >> 5);
      pSec->key_cnt = 0;
      pCc->can_device.length = 16;
      // Generate signature
      if (Ccuser_MakeSignature(&pCc->can_device,pSec,pCc->pkey_perm,&(pCc->pkey_perm[Cc_PERMKEY_LEN32-Cc_KEY_LEN32])))
      {
        pCc->state_group = CCgrpSTATE_IDLE;
        // inform application
        if ((pCc->grp_key_ctrl & Cc_GRPKEY_GENERATE) == 0)
        { // start of grouping process
          pCc->fctcb_event(Cc_EVENT_GROUP_INIT,pCc->grp_info_exp,(((UNSIGNED32) pCc->grp_HBevent) << 16) + pCc->grp_HBinhib);
        }
      }
      else
      { // signature not generated
        pCc->state_group = CCgrpSTATE_CLOSED;
      }
    }
    break;
  case CCgrpSTATE_IDLE:
    // now transmit the request
    pCc->fct_TxFIFO(&(pCc->can_device));
    // set timeout and next state
    pCc->grp_time = CCHW_GetTime() + Cc_SECHB_TIMEOUT;
    pCc->state_group = CCgrpSTATE_REQ_TX;
    break;
  case CCgrpSTATE_REQ_TX:
    if (pCc->grp_info_exp == pCc->grp_info_fnd)
    { // all expected nodes found and present
      pCc->grp_info_use = pCc->grp_info_fnd; // make this the current group
      pCc->can_device.dataByte[1] = pCc->my_status;
      pCc->can_device.dataByte[7] = 0; // no more device missing
      // now add security record
      pSec = (Cc_SECURITY_RECORD *) &(pCc->can_device.dataByte[8]);
      rnd = CCHW_Rand();
      pSec->rnd_lo = (UNSIGNED8) rnd;
      pSec->rnd_hi = (UNSIGNED8) (rnd >> 8);
      pSec->seq_lo = (UNSIGNED8) pCc->grp_hb_seq;
      pSec->seq_hi = (UNSIGNED8) (pCc->grp_hb_seq >> 8);
      pSec->pad_bytes = 0;
      pSec->encrypt = 0;
      pSec->active = 1;
      pSec->key_id = (pCc->grp_key_id_mm >> 5);
      // Generate signature
      if (Ccuser_MakeSignature(&pCc->can_device,pSec,pCc->pkey_perm,&(pCc->pkey_perm[Cc_PERMKEY_LEN32-Cc_KEY_LEN32])))
      {
        pCc->state_group = CCgrpSTATE_IDLE;
      }
      else
      { // signature not generated
        pCc->state_group = CCgrpSTATE_CLOSED;
      }      
      // re-transmit the request as confirmation
      pCc->fct_TxFIFO(&(pCc->can_device));
      pCc->state_group = CCgrpSTATE_HBINIT;
    }
    else
    { // still waiting for responses
      if (CCHW_IsTimeExpired(pCc->grp_time))
      { // timeout
        // here continue forever
        pCc->grp_time = CCHW_GetTime() + Cc_SECHB_TIMEOUT;
        // re-try grouping
        pCc->state_group = CCgrpSTATE_IDLE;
      }
    }
    break;
  case CCgrpSTATE_HBINIT:
    // Now create initial key from perm key and random values
    Ccuser_ExpandRandom(pCc->pkey_perm,rand_key,(UNSIGNED32 *) pCc->key_rnd);
    // If permanent key is bigger then dynamic key, vary portion of permanent key used to make key
    Ccuser_MakeKey(&(pCc->pkey_perm[rand_key[0] & (Cc_PERMKEY_LEN32-Cc_KEY_LEN32-1)]),rand_key,0,pCc->key_dyn);
#ifdef CANcrypt_DEBUG_DETAIL
    DebugOutKey(0xB0,(UNSIGNED32 *)pCc->key_rnd,pCc);
#endif
    // reset all counters
    mCc_ResetCounters(pCc,rand_key[Cc_KEY_LEN32-1]+rand_key[Cc_KEY_LEN32-2]);
    // delay for first cycle key generation: use half inhibit time
    pCc->grp_time = CCHW_GetTime() + (pCc->grp_HBinhib / 2);
    // special treatment during key generation
    if ((pCc->grp_key_ctrl & Cc_GRPKEY_GENERATE) != 0)
    { // starting a key generation cycle
      pCc->key_up_cnt = 0;
    }
    // start next secure HB cycle
    pCc->state_group = CCgrpSTATE_HBCYCLE1;
    break;
  case CCgrpSTATE_HBCYCLE1:
    // start of next HB cycle
    memset(&pCc->key_rnd,0,48); // clear previous random values
    rnd = CCHW_Rand(); // get new local random value
    // prepare collection of data, start with own
    pSec = (Cc_SECURITY_RECORD *) &(pCc->can_device.dataByte[4]);
    pSec->rnd_lo = (UNSIGNED8) rnd;
    pSec->rnd_hi = (UNSIGNED8) (rnd >> 8);
    pSec->seq_lo = (UNSIGNED8) pCc->grp_hb_seq;
    pSec->seq_hi = (UNSIGNED8) (pCc->grp_hb_seq >> 8);
    pSec->pad_bytes = 0;
    pSec->encrypt = 0;
    pSec->active = 1;
    pSec->key_id = Cc_DYN_KEY_GROUP;
    pSec->key_cnt = pCc->key_up_cnt;
    pCc->key_rnd[(pCc->my_address-1)*3] = (UNSIGNED8) rnd;
    pCc->key_rnd[((pCc->my_address-1)*3)+1] = (UNSIGNED8) (rnd >> 8);
    pCc->key_rnd[((pCc->my_address-1)*3)+2] = (UNSIGNED8) pSec->seq_lo;
    pCc->grp_info_fnd = (1 << pCc->my_address);
    pCc->can_device.dataByte[0] = Cc_CTRL_SECHB;
    pCc->can_device.dataByte[1] = pCc->my_status;
    pCc->can_device.dataByte[2] = 0;
    pCc->can_device.dataByte[3] = 0;
    pCc->can_device.length = 12;
    // Generate signature
    if (Ccuser_MakeSignature(&pCc->can_device,pSec,pCc->key_dyn,&(pCc->pkey_perm[Cc_PERMKEY_LEN32-Cc_KEY_LEN32])))
    {
      pCc->state_group = CCgrpSTATE_HBCYCLE2;
#if 0
      DebugOutKey(0xB1,(UNSIGNED32 *)pCc->key_rnd,pCc);
#endif
    }
    else
    { // signature not generated
      pCc->state_group = CCgrpSTATE_CLOSED;
    }
    break;
  case CCgrpSTATE_HBCYCLE2:
    // delay until first transmission
    if (CCHW_IsTimeExpired(pCc->grp_time))
    { // delay is over
      // transmit the request
      pCc->fct_TxFIFO(&(pCc->can_device));
      // Now set the time to collect responses
      pCc->grp_time = CCHW_GetTime() + pCc->grp_HBinhib;
      // Next state
      pCc->state_group = CCgrpSTATE_HBCYCLE3;
    }
    break;
  case CCgrpSTATE_HBCYCLE3:
    // Wait for all heartbeats
    if (pCc->grp_info_exp == pCc->grp_info_fnd)
    { // all nodes have answered
      // was there an ungroup request?
      if ( (pCc->grp_info_use == 0) ||
           ( ((pCc->grp_key_ctrl & Cc_GRPKEY_GENERATE) != 0) && (pCc->key_up_cnt >= 3) )
         ) // or end of key generation
      { // yes, ungroup
        if ( (pCc->grp_info_opt == Cc_EVENT_UNGROUP_SAVE) ||
             ( ((pCc->grp_key_ctrl & Cc_GRPKEY_GENERATE) != 0) && (pCc->key_up_cnt >= 3) )
           )
        { // save current key
          if ((pCc->grp_key_ctrl & Cc_GRPKEY_GENERATE) != 0)
          { // last key was a default key, replace
            Ccnvol_SavePermKey(pCc->grp_key_ctrl & 0x07,pCc->key_dyn);
            // use the temp generated permanent key as key ID
            Ccnvol_SavePermKeyID(pCc->grp_key_ctrl & 0x07,pCc->pkey_perm[0]);
            pCc->grp_key_ctrl = 0; // key control ends here
#if CANcrypt_DEBUG_DETAIL
            DebugOutKey(0xB4,(UNSIGNED32 *)pCc->key_dyn,pCc);
#endif
          }
          else
          { // save as session key
            Ccnvol_ErasePermKey(Cc_PERM_KEY_SESSION);
            Ccnvol_SavePermKey(Cc_PERM_KEY_SESSION,pCc->key_dyn);
            // use the temp generated permanent key as key ID
            Ccnvol_SavePermKeyID(Cc_PERM_KEY_SESSION,pCc->pkey_perm[0]);
#if CANcrypt_DEBUG_DETAIL
            DebugOutKey(0xB5,(UNSIGNED32 *)pCc->key_dyn,pCc);
#endif
          }
          // reset default key to last saved session key
          pCc->grp_key_id_mm = (Cc_PERM_KEY_SESSION << 5) + Cc_PERMKEY_LEN32;
          pCc->grp_key_ctrl = Cc_PERM_KEY_SESSION;
          pCc->fctcb_event(Cc_EVENT_UNGROUP_SAVE,pCc->my_address,0);
        }
        else
        {
          pCc->fctcb_event(Cc_EVENT_UNPAIR,pCc->my_address,0);
        }
        // update state and status
        pCc->state_group = CCgrpSTATE_CLOSED;
        pCc->my_status &= ~Cc_GROUP_STAT_BITS;
      }
      else
      { // copy last current dynamic key to last dynamic key
        memcpy(&(pCc->key_dyn[Cc_KEY_LEN32]),pCc->key_dyn,Cc_KEY_LEN8);
        // Now create next key from last key and random values
        Ccuser_ExpandRandom(pCc->key_dyn,rand_key,(UNSIGNED32 *) pCc->key_rnd);
        Ccuser_MakeKey(pCc->key_dyn,rand_key,0,pCc->key_dyn);
        pCc->key_up_cnt++;
        pCc->grp_hb_seq += Cc_SEC_INC;
#if CANcrypt_DEBUG_DETAIL
        DebugOutKey(0xD1,(UNSIGNED32 *)pCc->key_dyn,pCc);
#endif
        // start next cycle
        pCc->grp_time = CCHW_GetTime() + pCc->grp_HBevent;
        pCc->state_group = CCgrpSTATE_HBCYCLE1;
        if ((pCc->my_status & Cc_GROUP_STAT_BITS) == Cc_GROUP_STAT_PROGRESS)
        { // first call after grouping
          // change status to grouped
          pCc->my_status &= ~Cc_GROUP_STAT_BITS;
          pCc->my_status |= Cc_GROUP_STAT_GROUPED;
          if ((pCc->grp_key_ctrl & Cc_GRPKEY_GENERATE) == 0)
          { // only if not in key generation
            // minimize exposure of permanent keys, replace with current dynamic
            pCc->grp_key_ctrl |= (pCc->grp_key_id_mm >> 5);
            memcpy(pCc->pkey_perm,pCc->key_dyn,Cc_PERMKEY_LEN8);
            pCc->grp_key_id_mm = (Cc_PERM_KEY_SESSION << 5) + Cc_PERMKEY_LEN32;
            // inform application that we are grouped
            pCc->fctcb_event(Cc_EVENT_GROUPED,pCc->grp_info_exp,(((UNSIGNED32) pCc->grp_HBevent) << 16) + pCc->grp_HBinhib);
          }
        }
        else
        { // inform application that we are still secure
          if ((pCc->grp_key_ctrl & Cc_GRPKEY_GENERATE) == 0)
          { // only if not in key generation
            pCc->fctcb_event(Cc_EVENT_HBSECURED,pCc->grp_info_exp,(((UNSIGNED32) pCc->grp_HBevent) << 16) + pCc->grp_HBinhib);
          }
          else
          { // in key generation, use shorter timeout
            pCc->grp_time = CCHW_GetTime() + (pCc->grp_HBevent / 4);
          }
        }
      }
    }
    else
    { // not all required HB are in
      if (CCHW_IsTimeExpired(pCc->grp_time + Cc_SECHB_TIMEOUT))
      { // timout for secure HB receive is over
        pCc->err_counter = 128;
        pCc->state_group = CCgrpSTATE_CLOSED;
#ifdef CANcrypt_DEBUG_DETAIL
        DebugOutKey(0xBE,(UNSIGNED32 *) pCc->key_rnd,pCc);
#endif
        // Inform application
        pCc->fctcb_event(Cc_ERR_GROUP_LOST,pCc->grp_info_fnd,pCc->grp_info_exp);
      }
    }
    break;
  default:
    break;
  }
  
  // at any time, process unpairing command, if received
  if ( ((pCc->my_status & Cc_GROUP_STAT_BITS) == Cc_GROUP_STAT_GROUPED) &&
       (pCc->can_config.id >= Cc_CANID_CONFIG) && (pCc->can_config.id < Cc_CANID_CONFIG+15) &&
       (pCc->can_config.dataByte[0] == Cc_CTRL_UNPAIRGROUP)
     )
  { // unpair command
    pSec = (Cc_SECURITY_RECORD *) &(pCc->can_config.dataByte[4]);
    if (pSec->key_cnt == (pCc->key_up_cnt & 0x03))
    { // use current dynamic key
      verify = Ccuser_VerifySignature(&(pCc->can_config),pSec,pCc->key_dyn,&(pCc->pkey_perm[Cc_PERMKEY_LEN32-Cc_KEY_LEN32]));
    }
    else
    { // use last dynamic key
      verify = Ccuser_VerifySignature(&(pCc->can_config),pSec,&(pCc->key_dyn[Cc_KEY_LEN32]),&(pCc->pkey_perm[Cc_PERMKEY_LEN32-Cc_KEY_LEN32]));
    }
    if (verify)
    { // checksum verified
      // send response
      Cc_TxDisconnect(pCc,pCc->can_config.id & 0x0F,pCc->can_config.dataByte[2]);
      if (pCc->err_counter > 0)
      {
        pCc->err_counter--;
      }
#ifdef CANcrypt_DEBUG_DETAIL
      DebugOutKey(0xED,NULL,pCc);
#endif
    }
    else
    {
      pCc->err_counter += 63;
#ifdef CANcrypt_DEBUG_DETAIL
      DebugOutKey(pSec->seq_lo,NULL,pCc);
#endif
    }
    pCc->can_config.dataByte[0] = Cc_CTRL_ABORT; // do not come back here
    ret_val = TRUE;
  }

  return ret_val;
}


/**************************************************************************
DOES:    This is the CAN receive function of CANcrypt, must be called
         with every message received, directly from CAN receive interrupt.
HERE:    Handles Grouping messages and secure Heartbeat
RETURNS: TRUE, if this message was processed by CANcrypt
         FALSE, if this message was ignored by CANcrypt
**************************************************************************/
UNSIGNED8 Cc_Process_Group_Rx(
  Cc_HANDLE *pCc,       // Pointer to handle with CANcrypt data
  CAN_MSG *pCANrx       // Pointer to CAN message received
)
{
UNSIGNED8 device;
UNSIGNED8 ret_val = FALSE;
Cc_SECURITY_RECORD *pSec; // CANcryprt security record

  switch(pCc->state_group)
  {
  case CCgrpSTATE_REQ_TX:
    // in this state, collecting all grouping requests
    if ( (pCANrx->id >= Cc_CANID_CONFIG) && (pCANrx->id < Cc_CANID_CONFIG+15) &&
         (pCANrx->dataByte[0] == Cc_CTRL_PAIRGROUP)
       )
    { // received CANcrypt device message, grouping request
      device = pCANrx->id - Cc_CANID_CONFIG + 1;
      pSec = (Cc_SECURITY_RECORD *) &(pCANrx->dataByte[8]);
      if (Ccuser_VerifySignature(pCANrx,pSec,pCc->pkey_perm,&(pCc->pkey_perm[Cc_PERMKEY_LEN32-Cc_KEY_LEN32])))
      { // checksum verified
        // mark this as received
        pCc->grp_info_fnd |= (1 << device);
        // retrieve the random value
        pCc->key_rnd[ (device-1)*3   ] = pCANrx->dataByte[4];
        pCc->key_rnd[((device-1)*3)+1] = pCANrx->dataByte[5];
        pCc->key_rnd[((device-1)*3)+2] = pCANrx->dataByte[6];
        // if all found, then we are done
        if (pCc->grp_info_exp == pCc->grp_info_fnd)
        { // all mandatory nodes found and present
          if (pCc->grp_info_opt == 0) 
          { // no optional nodes
            // timer expired now
            pCc->grp_time = CCHW_GetTime();
          }
          else
          { // give optional nodes a bit longer to show up
            pCc->grp_time = CCHW_GetTime() + pCc->cycle_timeout;
          }
        }
      }
      else
      { // Grouping request does not match
        if ((pCANrx->dataByte[2] & 0x0F) > (pCc->grp_key_id_mm >> 5))
        { // Inform application, a higher key ID is requested
          // Lower ID are handled automatically
          Cc_QueueEvent(pCc,Cc_ERR_GROUP_KEYID,pCANrx->dataByte[2],pCANrx->dataByte[3]);
        }
        else if ( ((pCANrx->dataByte[2] & Cc_GRPKEY_GENERATE) != 0) &&
                  ((pCc->grp_key_ctrl & Cc_GRPKEY_GENERATE) == 0)
                )
        { // key generation request came in, inform app, if not initated by us
          Cc_QueueEvent(pCc,Cc_ERR_GROUP_KEYGEN_REQ,pCANrx->dataByte[2] & 0x0F,pCANrx->dataByte[3]);
        }
      }
      // mark as processed
      ret_val = TRUE;
    }
    break;
  case CCgrpSTATE_HBCYCLE3:
  case CCgrpSTATE_HBCYCLE2:
    // in this state, collecting all secure heartbeats
    if ( (pCANrx->id >= Cc_CANID_CONFIG) && (pCANrx->id < Cc_CANID_CONFIG+15) &&
         (pCANrx->dataByte[0] == Cc_CTRL_SECHB)
       )
    { // received CANcrypt device message, secure heartbeat
      device = pCANrx->id - Cc_CANID_CONFIG + 1;
      pSec = (Cc_SECURITY_RECORD *) &(pCANrx->dataByte[4]);
      if ( (pSec->seq_lo == (UNSIGNED8) pCc->grp_hb_seq) && (pSec->seq_hi == (UNSIGNED8)(pCc->grp_hb_seq >> 8)) && 
           (Ccuser_VerifySignature(pCANrx,pSec,pCc->key_dyn,&(pCc->pkey_perm[Cc_PERMKEY_LEN32-Cc_KEY_LEN32])))
         )
      { // checksum verified
        // mark this as received
        pCc->grp_info_fnd |= (1 << device);
        // retrieve the random value
        pCc->key_rnd[ (device-1)*3   ] = pSec->rnd_lo;
        pCc->key_rnd[((device-1)*3)+1] = pSec->rnd_hi;
        pCc->key_rnd[((device-1)*3)+2] = pSec->seq_lo;
        // if all found, then we are done
        if (pCc->grp_info_exp == (pCc->grp_info_exp & pCc->grp_info_fnd))
        { // all mandatory nodes found and present
          // ensure timer expired
          pCc->grp_time = CCHW_GetTime();
        }
        if (pCc->err_counter > 0)
        {
          pCc->err_counter--;
        }
        // mark as processed
        ret_val = TRUE;
      }
      else
      { // secure heartbeat failure
#ifdef CANcrypt_DEBUG_DETAIL
        DebugOutKey(0x6E,NULL,pCc);
#endif   
        pCc->state_group = CCgrpSTATE_CLOSED;
        if ((pCc->my_status & Cc_GROUP_STAT_BITS) == Cc_GROUP_STAT_PROGRESS)
        { // first call after grouping
          // clean status of grouped
          pCc->my_status &= ~Cc_GROUP_STAT_BITS;
          // inform application that grouping failed
          pCc->fctcb_event(Cc_ERR_GROUP_KEYFAIL,pCc->grp_key_id_mm >> 5,pCc->grp_key_id_mm & 0x7F);
        }
        else
        { // Inform application
          Cc_QueueEvent(pCc,Cc_ERR_SECHB_FAIL,device,0);
        }
      }
    }
    break;
  default:
    break;
  }
  
  // at any time, process ungrouping command, 
  // if grouped and broadcast
  if ( ((pCc->my_status & Cc_GROUP_STAT_BITS) == Cc_GROUP_STAT_GROUPED) &&
       (pCANrx->id >= Cc_CANID_CONFIG) && (pCANrx->id < Cc_CANID_CONFIG+15) &&
       (pCANrx->dataByte[0] == Cc_CTRL_UNPAIRGROUP)
     )
  { // unpair command
    // copy message for later access
    memcpy(&(pCc->can_config),pCANrx,sizeof(CAN_MSG));
    // mark as processed
    ret_val = TRUE;
  }

  return ret_val;
}

#endif

#ifdef __cplusplus
}
#endif
/*----------------------- END OF FILE ----------------------------------*/
