/********************************************************************
MODULE:    CANcrypt_userfct.h, customizable functions
CONTAINS:  Customizable security functions of CANcryot
AUTHORS:   Embedded Systems Academy, Inc (USA) and
           Embedded Systems Academy, GmbH (Germany)
HOME:      https://www.cancrypt.net
LICENSE:   See below, APPLIES TO THIS FILE ONLY. 
           See individual file headers for applicable license.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at
  www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or 
  implied. See the License for the specific language governing 
  permissions and limitations under the License.

VERSION:   1.03, 06-AUG-2018
           $LastChangedRevision: 466 $
********************************************************************/


#ifndef _CANCRYPT_USERFCT_H
#define _CANCRYPT_USERFCT_H

#include "CANcrypt_types.h"


/********************************************************************
BOOK:    Section 6.1 "Collect random numbers"
DOES:    This function expands an array with a limited number of 
         random bytes to an array of random bytes with the length
         of the current dynamic key.
RETURNS: nothing
********************************************************************/
void Ccuser_ExpandRandom(
  UNSIGNED32 *pkey,     // key input, length Cc_PERMKEY_LEN_BITS
  UNSIGNED32 *pdest,    // destination: array with length of dyn. key
  UNSIGNED32 *psrc      // array with zeros and random numbers (3*15)
);


/********************************************************************
BOOK:    Section 6.2 "Bit mixup"
DOES:    This function mixes the bits in a 64bit value by applying
         a Speck cipher. Used by key initialization functions and
         one-time pad generation.
MOTE:    Recommended number of rounds is 27
RETURNS: Value pmixed[] returns the mixed bits
********************************************************************/
void Ccuser_Mix64( 
  UNSIGNED32 pkey[2],   // key input of 64 bit
  UNSIGNED32 pdat[2],   // data input of 64 bit
  UNSIGNED32 pmixed[2], // mixed bits output of 64 bit
  UNSIGNED8 rounds      // number of mixing rounds to execute
);


/********************************************************************
BOOK:    Section 6.3 "Generate keys"
DOES:    Takes input from 2 keys and 1 factor to create a new key.
         Used to create a dynamic key from a permanent key using 
         random input and a serial number.
         Used to create a one-time pad from a permanent and 
         dynamic key and a counter
RETURNS: TRUE if key initialization completed, 
         FALSE if not possible due to parameters
********************************************************************/
UNSIGNED8 Ccuser_MakeKey(
  UNSIGNED32 *pin1,     // input 1: pointer to permanent key used
  UNSIGNED32 *pin2,     // input 2: pointer to 2nd input array
  UNSIGNED32 factor,    // input 3: optional, set zero if not used
                        // used for serial number, counter
  UNSIGNED32 *pout      // output: the dynamic key or one time pad
);


/********************************************************************
BOOK:    Section 6.4.1 "Pairing: Key update using a single bit"
DOES:    Called to update a dynamic key by introducing a new bit.
RETURNS: TRUE if key update completed, 
         FALSE if not possible due to parameters
********************************************************************/
UNSIGNED8 Ccuser_UpdateDynKey(
  UNSIGNED8 bit,        // new bit to introduce to dynamic key
  UNSIGNED32 *ppermkey, // pointer to permanent key used
  UNSIGNED32 *pdynkey   // pointer to dynamic key
);


/********************************************************************
BOOK:    Section 6.5.1 "Generate signature value"
         CANCRYPT FD ADOPTED VERSION
         Modified for CANcrypt FD, signs entire CAN FD message
DOES:    Generates a signature for a CAN FD message
RETURNS: TRUE, when generation success
********************************************************************/
UNSIGNED8 Ccuser_MakeSignature(
  CAN_MSG *pCAN,        // CAN FD message to generate signature for
  Cc_SECURITY_RECORD *pSec, // Security record, values pre-filled 
  UNSIGNED32 pKey[Cc_KEY_LEN32], // key used for signature
  UNSIGNED32 pVec[Cc_KEY_LEN32]  // key used for chksum init
);


/********************************************************************
BOOK:    Section 6.5.2 "Verify signature value"
         CANCRYPT FD ADOPTED VERSION
DOES:    Verifies a signature received in a CAN FD message
RETURNS: TRUE, if signature was verified
********************************************************************/
UNSIGNED8 Ccuser_VerifySignature(
  CAN_MSG *pCAN,        // CAN FD message to generate signature for
  Cc_SECURITY_RECORD *pSec, // Security record, values pre-filled 
  UNSIGNED32 pKey[Cc_KEY_LEN32], // key used for signature
  UNSIGNED32 pVec[Cc_KEY_LEN32]  // key used for chksum init
);


/********************************************************************
BOOK:    Section 6.6.1 "Checksum init"
DOES:    Generates an initial value for the checksum, depending on
         a key passed.
RETURNS: 32 bit initial value
********************************************************************/
UNSIGNED32 Ccuser_ChecksumInit(
  UNSIGNED32 *pkey      // pointer to a key (pad, dyn or perm)
);


/********************************************************************
BOOK:    Section 6.6.2 "Checksum step"
DOES:    Calculates a 16bit checksum, adding one value at the time
RETURNS: Checksum value in lowest 16bit, highest 16bit is carry-over
********************************************************************/
UNSIGNED32 Ccuser_ChecksumStep16(
  UNSIGNED32 last,      // initial value or last calculated value
                        // higher 16bit may include a carry-over
  UNSIGNED16 *pdat      // next 16bit value to add
);


/********************************************************************
BOOK:    Section 6.6.3 "Checksum final"
DOES:    When checksum calculation is completed, merges 16bit 
         checksum with 16bit carry ove rto final 16bit checksum.
RETURNS: Final checksum value
********************************************************************/
UNSIGNED16 Ccuser_ChecksumFinal(
  UNSIGNED32 last       // last calculated checksum value
);


/********************************************************************
BOOK:    Section 6.7.1 "Secure message encryption"
DOES:    Encrypts a data block in a secure message
NOTE:    This version NOT optimized for 32 bit architecture
RETURNS: TRUE if encryption completed, 
         FALSE if not possible due to parameters
********************************************************************/
UNSIGNED8 Ccuser_Encrypt(
  UNSIGNED32 *ppad,     // pointer to current one-time pad
  UNSIGNED32 *pdat,     // pointer to the data to encrypt
  UNSIGNED16 first,     // first byte to encrypt
  UNSIGNED16 bytes      // number of bytes to encrypt
);


/********************************************************************
BOOK:    Section 6.7.2 "Secure message decryption"
NOTE:    Only used if cryptographic function is not symmetric and 
         decryption requires a different function then encryption
DOES:    Decrypts a data block 
RETURNS: TRUE if decryption completed, 
         FALSE if not possible due to parameters
********************************************************************/
UNSIGNED8 Ccuser_Decrypt(
  UNSIGNED32 *ppad,     // pointer to current one-time pad
  UNSIGNED32 *pdat,     // pointer to the data to decrypt
  UNSIGNED16 first,     // first byte to decrypt
  UNSIGNED16 bytes      // number of bytes to decrypt
);

#endif
/*----------------------- END OF FILE -----------------------------*/
