/********************************************************************
MODULE:    CANcrypt_types.h, global type definitions
CONTAINS:  The global type definitions required for CANcrypt
AUTHORS:   Embedded Systems Academy, Inc (USA) and
           Embedded Systems Academy, GmbH (Germany)
HOME:      https://www.cancrypt.net
LICENSE:   See below, APPLIES TO THIS FILE ONLY. 
           See individual file headers for applicable license.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at
  www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or 
  implied. See the License for the specific language governing 
  permissions and limitations under the License.

VERSION:   1.03, 06-AUG-2018
           $LastChangedRevision: 466 $
********************************************************************/


#ifndef _CANCRYPT_TYPES_H
#define _CANCRYPT_TYPES_H

#include "Cc_user_types.h"
#include "Cc_user_config.h"
#include "CANcrypt_can.h"


/********************************************************************
CANcrypt FD 16bit version info
6bit version, 6bit revision, 2bit reserved (zero), 2bit functionality
Version zero reserved for demo and evaluation
********************************************************************/
// Version 0.1
#define Cc_VERSION_NR  0
#define Cc_REVISION_NR 1
#define CANcrypt_VERSION ( (Cc_VERSION_NR << 10) + \
                           (Cc_REVISION_NR << 4) + Cc_FUNCTIONALITY )


/********************************************************************
CANcrypt key hierarchy
Book section 5.3.2 "The Keys"
********************************************************************/
#define Cc_PERM_KEY_MANUFACTURER  0x06
#define Cc_PERM_KEY_INTEGRATOR    0x05
#define Cc_PERM_KEY_OWNER         0x04
#define Cc_PERM_KEY_USER          0x03
#define Cc_PERM_KEY_SESSION       0x02
#define Cc_DYN_KEY_GROUP          0x01
#define Cc_DYN_KEY_PAIR           0x00
  
  
/********************************************************************
CANcrypt status byte
Book section 5.3.3 "Status"
********************************************************************/
// Bits 0-1: status of pairing process
//#define Cc_PAIR_STAT_BITS         0x03
//#define Cc_PAIR_STAT_NONE         0x00
//#define Cc_PAIR_STAT_PROGRESS     0x01
//#define Cc_PAIR_STAT_PAIRED       0x02
//#define Cc_PAIR_STAT_FAIL         0x03
// Bits 2-3: status of grouping process
#define Cc_GROUP_STAT_BITS        0x0C
#define Cc_GROUP_STAT_NONE        0x00
#define Cc_GROUP_STAT_PROGRESS    0x04
#define Cc_GROUP_STAT_GROUPED     0x08
#define Cc_GROUP_STAT_FAIL        0x0C
// Bits 4-5: status of last command
#define Cc_CMD_STAT_BITS          0x30
#define Cc_CMD_STAT_NONE          0x00
#define Cc_CMD_STAT_SUCCESS       0x10
#define Cc_CMD_STAT_IGNORE        0x20
#define Cc_CMD_STAT_FAIL          0x30
// Bit 6: reserved
// Bit 7: status of key generation
#define Cc_KEY_STAT_BITS          0x80
#define Cc_KEY_STAT_NONE          0x00
#define Cc_KEY_STAT_GENERATION    0x80


/********************************************************************
CANcrypt request, response, event identification
Book section 5.3.4 "Controls"
Only partial
********************************************************************/
//#define Cc_CTRL_BITS              0x0F
#define Cc_CTRL_ABORT             0x00
#define Cc_CTRL_ACK               0x01
#define Cc_CTRL_ALERT             0x02
#define Cc_CTRL_IDENTIFY          0x03
#define Cc_CTRL_PAIRGROUP         0x04
#define Cc_CTRL_UNPAIRGROUP       0x05
//#define Cc_CTRL_FLIPBIT           0x06
//#define Cc_CTRL_KEYGEN            0x07
//#define Cc_CTRL_BITNOW            0x08
#define Cc_CTRL_SECHB             0x09
//#define Cc_CTRL_DATRD             0x0A
//#define Cc_CTRL_DATWR             0x0B
//#define Cc_CTRL_DATMSG            0x0C
//#define Cc_CTRL_XTDID             0x0D
#define Cc_CTRL_RESERVED          0x0E
#define Cc_CTRL_SAVEKEY           0x0F


/********************************************************************
CANcrypt security functionality
Book section 5.3.5 "Methods"
********************************************************************/
// Security functionality: basic, regular, advanced or custom
#define Cc_SECFCT_BITS            0x03
#define Cc_SECFCT_BASIC           0x00
#define Cc_SECFCT_REGULAR         0x01
#define Cc_SECFCT_ADVANCED        0x02
#define Cc_SECFCT_CUSTOM          0x03
// Bit generation method: random delay or direct response
#define Cc_BITMETHOD_DIRECT       0x00
#define Cc_BITMETHOD_DELAY        0x04
// Bit claiming messages: 2 or 16
#define Cc_BITMETHOD_2CLAIMS      0x00
#define Cc_BITMETHOD_16CLAIMS     0x08


/********************************************************************
CANcrypt timings and timeouts
Book section 5.3.7 "Timings"
********************************************************************/
#define Cc_TIMING_BITS            0x03
#define Cc_TIMING_FAST            0x00
#define Cc_TIMING_MEDIUM          0x01
#define Cc_TIMING_SLOW            0x02
#define Cc_TIMING_CUSTOM          0x03


/********************************************************************
CANcrypt call back event and alert / error codes
Below 0x80: status, event  -  0x80 and higher: error, alert
********************************************************************/
#define Cc_EVENT_GENSTAT          0x00
#define Cc_EVENT_KEYSTAT          0x10
#define Cc_EVENT_PARSTAT          0x20
#define Cc_EVENT_GRPSTAT          0x30
#define Cc_EVENT_MSGSTAT          0x40
#define Cc_EVENT_GASTAT           0x50
#define Cc_EVENT_IDENT            0x60
#define Cc_EVENT_GENERR           0x80
#define Cc_EVENT_KEYERR           0x90
#define Cc_EVENT_PARERR           0xA0
#define Cc_EVENT_GRPERR           0xB0
#define Cc_EVENT_MSGERR           0xC0
#define Cc_EVENT_GAERR            0xD0

// 0x0X: unexpected ACK, signed
// unknown ACK, currently unused
#define Cc_EVENT_GENACK           (Cc_EVENT_GENSTAT + 1)
// ungrouping event with instruction to 
// save current dynamic key as session key
// param1: own device ID, param2: 0
#define Cc_EVENT_UNGROUP_SAVE     (Cc_EVENT_GENSTAT + 2)
// ungrouping / unpairing without saving keys
// param1: own device ID, param2: 0
#define Cc_EVENT_UNPAIR           (Cc_EVENT_GENSTAT + 3)
#define Cc_EVENT_UNGROUP          (Cc_EVENT_GENSTAT + 3)

// 0x1X: new key generation initated
// unsued
#define Cc_EVENT_KEYGEN_INIT      (Cc_EVENT_KEYSTAT + 1)
// key generation completed OK
// param1: key generated, param2.0-7: bit pos, param2:8-15: key ID (major/minor)
#define Cc_EVENT_KEYGEN_OK        (Cc_EVENT_KEYSTAT + 2)
// key generation requested flip bits
// param1: key generated, param2.0-7: bit pos, param2:8-15: key ID (major/minor)
#define Cc_EVENT_KEYGEN_FLIP      (Cc_EVENT_KEYSTAT + 3)
// key generation flip bit acknowledge
// currently unused
#define Cc_EVENT_KEYGEN_FLIP_ACK  (Cc_EVENT_KEYSTAT + 4)
// key generation completed, save key
// param1: key major id, param2: 0
#define Cc_EVENT_KEYGEN_SAVE_KEY  (Cc_EVENT_KEYSTAT + 5)
// single bit genertaed ok for dyn key update
// param1: key info major/minor, param2: bit generated
#define Cc_EVENT_BITGEN_OK        (Cc_EVENT_KEYSTAT + 6)

// 0x2X: new pairing started
// currently unused
#define Cc_EVENT_PAIR_INIT        (Cc_EVENT_PARSTAT + 1)
// now pairing is active
// param1: 0, param2: 0
#define Cc_EVENT_PAIRED           (Cc_EVENT_PARSTAT + 2)

// 0x3X: new grouping cycle
// param1: group info, param2.0-15: HB inhibit time, param2.16-31: HB event time
#define Cc_EVENT_GROUP_INIT       (Cc_EVENT_GRPSTAT + 1)
// grouping completed
// param1: group info, param2.0-15: HB inhibit time, param2.16-31: HB event time
#define Cc_EVENT_GROUPED          (Cc_EVENT_GRPSTAT + 2)
// secure heartbeat, single cycle completed
// param1: group info, param2.0-15: HB inhibit time, param2.16-31: HB event time
#define Cc_EVENT_HBSECURED        (Cc_EVENT_GRPSTAT + 3)

// 0x5X: generic read access requested (device only)
// currently unused, own generic access call back used instead
#define Cc_GACC_READ_REQUEST      (Cc_EVENT_GASTAT + 1)
// generic read access completed (configurator only)
// param1: data length, param2: data
#define Cc_GACC_READ_OK           (Cc_EVENT_GASTAT + 2)
// generic write access requested (device only)
// currently unused, own generic access call back used instead
#define Cc_GACC_WRITE_REQUEST     (Cc_EVENT_GASTAT + 3)
// generic write access completed ok (configurator only)
// param1: 0, param2: 0 
#define Cc_GACC_WRITE_OK          (Cc_EVENT_GASTAT + 4)
// generic write access completed, but write failed (configurator only)
// param1: lenth requested, param2: length written
#define Cc_GACC_WRITE_NAK         (Cc_EVENT_GASTAT + 5)

// 0x6X: Configurator: identification response received
// param1.0-15:key id and size, param1.16-19: device id (bits 16-19) 
// param2: version and method
#define Cc_CONFIG_IDENT           (Cc_EVENT_IDENT + 1)
// param1: length, param2: data
#define Cc_CONFIG_XTDIDENT        (Cc_EVENT_IDENT + 2)


// 0x8X: general errors
// unexpected ACK, failed signature
// currently unused
#define Cc_EVENT_GENACKFAIL       (Cc_EVENT_GENERR + 1)
// intruder alert
// injected message detected
#define Cc_ERR_INTRUDER           (Cc_EVENT_GENERR + 2)
// error counter overflow
// param1: 0, param2: 0
#define Cc_ERR_ERRCOUNT           (Cc_EVENT_GENERR + 3)
// signature of last request does not match
// param1: own status, param2:0
#define Cc_ERR_ABORT              (Cc_EVENT_GENERR + 4)
// abort from communication partner received
// para1: partner device ID, param2: 0
#define Cc_ERR_ABORT_PAR          (Cc_EVENT_GENERR + 5)
// identify abort, requested service not available
// param1: CANcrypt version, param2: method & timing requested
#define Cc_ERR_IDENT              (Cc_EVENT_GENERR + 6)
// Rx call-back event queue overrun, events lost
// param1: unknown, param2: unknown
#define Cc_ERR_CBE_OVERRUN        (Cc_EVENT_GENERR + 7)
// Security delay on errors activated
// currently unused
#define Cc_ERR_SEC_DELAY          (Cc_EVENT_GENERR + 8)

// 0x9X: error in key generation
// timeout, key generation not completed
// param1: key generated, param2.0-7: bit pos, param2:8-15: key ID (major/minor)
#define Cc_ERR_KEY_GEN_TIME       (Cc_EVENT_KEYERR + 1)
// different bit expected in key generation
// param1: key generated, param2.0-7: bit pos, param2:8-15: key ID (major/minor)
#define Cc_ERR_KEY_GEN_BIT        (Cc_EVENT_KEYERR + 2)
// too many bits had to be ignored, generation aborted
// param1: key generated, param2.0-7: bit pos, param2:8-15: key ID (major/minor)
#define Cc_ERR_KEY_GEN_IGNORE     (Cc_EVENT_KEYERR + 3)
// signature failed
// param1: key used, param2.0-7: bit pos, param2:8-15: key ID (major/minor)
#define Cc_ERR_KEY_GEN_SIGFAIL    (Cc_EVENT_KEYERR + 4)
// Requested key not available
// param1: key major, param2: key minor
#define Cc_ERR_KEY_NO_KEY         (Cc_EVENT_KEYERR + 5)
// flip request timeout
// param1: last flip value, param2.0-7: bit pos, param2:8-15: key ID (major/minor)
#define Cc_ERR_KEY_FLIP_TIMEOUT   (Cc_EVENT_KEYERR + 6)
// flip request failed
// param1: last flip value, param2.0-7: bit pos, param2:8-15: key ID (major/minor)
#define Cc_ERR_KEY_FLIP_FAIL      (Cc_EVENT_KEYERR + 7)
// key generation: abort received
// param1: 0, param2: 0
#define Cc_ERR_KEY_ABORT          (Cc_EVENT_KEYERR + 8)
// Key generation method requested not available
// param1.0-7: key id, param1.8-15: key len, param2: method and timing 
#define Cc_ERR_KEY_METHOD         (Cc_EVENT_KEYERR + 9)

// 0xAX: pairing error
// trying to group and pair at the same time
// currently unused
#define Cc_ERR_PAIRGOUPED         (Cc_EVENT_PARERR + 1)
// pairing lost
// param1: 0, param2: 0
#define Cc_ERR_PAIR_LOST          (Cc_EVENT_PARERR + 2)
// pairing failed on wrong key
// param1: own status, param2: 0
#define Cc_ERR_PAIR_FAIL          (Cc_EVENT_PARERR + 3)

// 0xBX: grouping error
// grouping failed before secure heartbeat started
// currently unused, group does not match
#define Cc_ERR_GROUP_SESS         (Cc_EVENT_GRPERR + 1)
// group partner lost
// param1: group expected, param2: group found
#define Cc_ERR_GROUP_LOST         (Cc_EVENT_GRPERR + 2)
// secure heartbeat failure
// param1: device with sec HB failure, param2: 0
#define Cc_ERR_SECHB_FAIL         (Cc_EVENT_GRPERR + 3)
// different key requested, key ID and size
#define Cc_ERR_GROUP_KEYID        (Cc_EVENT_GRPERR + 4)
// grouping failed, wrong key, key ID and size
#define Cc_ERR_GROUP_KEYFAIL      (Cc_EVENT_GRPERR + 5)
// key generation, key ID and size
#define Cc_ERR_GROUP_KEYGEN       (Cc_EVENT_GRPERR + 6)
// key generation request, key ID and size
#define Cc_ERR_GROUP_KEYGEN_REQ   (Cc_EVENT_GRPERR + 7)
// key can not be generated (NVOL used), key ID and size
#define Cc_ERR_GROUP_KEYNOGEN     (Cc_EVENT_GRPERR + 8)

// 0xCX: secure messaging error
// unexpected message received
// param1: number of unexpected messages, param2: message counter
#define Cc_ERR_MSG_UNEXPECT       (Cc_EVENT_MSGERR + 1)
// secure message failure, not paired
// param1: CAN ID used, param2: unknown
#define Cc_ERR_MSG_NOPAIR         (Cc_EVENT_MSGERR + 2)
// secure message failure, not secure
// param1: CAN ID used, param2: unknown
#define Cc_ERR_MSG_NOSEC          (Cc_EVENT_MSGERR + 3)

// 0xDX: generic read/write error
// generic read access failed
// currently unused
#define Cc_GACC_READ_FAIL         (Cc_EVENT_GAERR + 1)
// generic read access timeout
// currently unused
#define Cc_GACC_READ_TIME         (Cc_EVENT_GAERR + 2)
// generic write access failed
// currently unused
#define Cc_GACC_WRITE_FAIL        (Cc_EVENT_GAERR + 3)
// generic write access timeout
// currently unused
#define Cc_GACC_WRITE_TIME        (Cc_EVENT_GAERR + 4)


/********************************************************************
CANcrypt control for Cc_Restart
********************************************************************/
// Bits 0-1: control for pairing process
//#define Cc_PAIR_CTRL_NONE         0x00
//#define Cc_PAIR_CTRL_RESTART      0x01
//#define Cc_PAIR_CTRL_STOP         0x02
// Bits 2-3: control for grouping process
#define Cc_GROUP_CTRL_NONE        0x00
#define Cc_GROUP_CTRL_RESTART     0x04
#define Cc_GROUP_CTRL_STOP        0x08


/********************************************************************
CANcrypt control for Cc_SelectKey
********************************************************************/
// Initiate key generation / grouping for this key
#define Cc_GRPKEY_GENERATE        0x80


/********************************************************************
CANcrypt FD Security Record.
Split in lo and hi bytes to be independant of endianess.
********************************************************************/
typedef struct {
  UNSIGNED8 rnd_lo;       // random value
  UNSIGNED8 rnd_hi;
  UNSIGNED8 seq_lo;       // sequence tracking
  UNSIGNED8 seq_hi;
  UNSIGNED8 pad_bytes: 4; // number of padding bytes
  UNSIGNED8 reserved1: 2;
  UNSIGNED8 encrypt:   1; // data in front of record is encrypted
  UNSIGNED8 active:    1; // this device particpats in key update
  UNSIGNED8 key_id:    4; // key used for signature
  UNSIGNED8 key_cnt:   2; // key update counter
  UNSIGNED8 reserved2: 2;
  UNSIGNED8 sign_lo;      // signature
  UNSIGNED8 sign_hi;
} Cc_SECURITY_RECORD;


#ifdef Cc_USE_MONITORING
/********************************************************************
CAN message monitoring, list of CAN IDs not expected for receive
********************************************************************/
typedef struct {
 #if (Cc_CANIDLIST_LEN != 0)
  COBID_TYPE   lst[Cc_CANIDLIST_LEN]; // list of CAN IDs
 #endif
  COBID_TYPE   self_rx;     // last tx with self receive
  COBID_TYPE   unx_alert;   // unx message received
 #if (Cc_CANIDLIST_LEN != 0)
  UNSIGNED16   cur_num;     // current number of entries in list
 #endif
} Cc_CANIDLIST_HANDLE;
#endif


#ifdef Cc_USE_SECURE_MSG
/********************************************************************
Book section 5.4 "CANcrypt secure message table"
********************************************************************/
typedef struct {
  COBID_TYPE  CAN_ID_Match; // CAN message ID of the secure message
  COBID_TYPE  CAN_ID_Mask;  // Mask used on Rx CAN ID
                            // (CAN_Rx_id & CAN_ID_Mask) == CAN_ID_Match)
  UNSIGNED8   EncryptFirst; // first byte to encrypt
  UNSIGNED8   EncryptLen;   // number of bytes to encrypt
  UNSIGNED8   FunctMethod;  // function and methods
  UNSIGNED8   Producer;     // device/node ID of the producer
} Cc_SEC_MSG_TABLE_ENTRY;
#endif

#ifdef Cc_USE_PAIRING
/********************************************************************
CANcrypt internal state machine for key generation
********************************************************************/
typedef enum 
{
  CCkeySTATE_IDLE,          // no action, waiting for command
  CCkeySTATE_DYN_KEY_INIT,  // start initialization of dynamic key
  CCkeySTATE_START_KEY_REQ, // configurator: start key generation request
  CCkeySTATE_NEXT,          // now activating next random bit cycle
  CCkeySTATE_BITCYCLE,      // bit cycle now active
  CCkeySTATE_GOT_BIT,       // a bit for key is determined
  CCkeySTATE_ADD_BIT,       // the bit is added to key
  CCkeySTATE_IGNORE_BIT,    // no bit determined, ignore
  CCkeySTATE_DELAY_NEXT,    // configurator: delay until next cycle
  CCkeySTATE_OVER_OK,       // all random bit cycles completed, key ok
  CCkeySTATE_OVER_DELAY,    // delay after completion, before ACK and CB
  CCkeySTATE_CLOSED,        // channel closed not using CANcrypt anymore
  CckeySTATE_ABORT          // abort received during key generation
} CCkeySTATE_TYPE;


/********************************************************************
CANcrypt internal state machine for pairing
********************************************************************/
typedef enum 
{
  CCparSTATE_IDLE,          // not paired, wait for pairing action
#if (Cc_DEVICE_ID == 1)
  CCparSTATE_WAIT_RESP,     // waiting for pairing response
  CCparSTATE_RESPONSE,      // pairing response received, sending ack
  CCparSTATE_WAIT_ACK,      // wait for ack
#else
  CCparSTATE_REQ_RX,        // initial pairing request transmitted
#endif
  CCparSTATE_KEYUPDATE,     // Updating key with bit cycles
  CCparSTATE_ACK,           // send acknowledge
  CCparSTATE_PRE_PAIRED,    // delay to start (wait for partner ack)
  CCparSTATE_PAIRED,        // paired 
  CCparSTATE_ABORT,         // aborted
  CCparSTATE_CLOSED         // pairing closed, do not re-try
} CCparSTATE_TYPE;
#endif


#ifdef Cc_USE_GROUPING
/********************************************************************
CANcrypt internal state machine for grouping
********************************************************************/
typedef enum 
{
  CCgrpSTATE_INIT,          // first initialization
  CCgrpSTATE_IDLE,          // not grouped, try to pair
  CCgrpSTATE_REQ_TX,        // initial grouping request transmitted
  CCgrpSTATE_HBINIT,        // grouping completed, init secure HB
  CCgrpSTATE_HBCYCLE1,
  CCgrpSTATE_HBCYCLE2,      // 3 states fpr each cycle
  CCgrpSTATE_HBCYCLE3,
  CCgrpSTATE_CLOSED         // grouping closed
} CCgrpSTATE_TYPE;
#endif


/********************************************************************
DOES:    Call-back function, reports a system status change to the 
         application
RETURNS: Nothing
********************************************************************/
typedef void (*Cc_CB_EVENT) (
  UNSIGNED8   event,        // event that happened (Cc_EVENT_xxx)
  UNSIGNED32  param1,       // depending on the event, up to 3 
  UNSIGNED32  param2        // event related parameters
);


/********************************************************************
DOES:    Call-back function to pass on a CAN message to a buffer
RETURNS: TRUE, if message was accepted
         FALSE, if message could not be processed
********************************************************************/
typedef UNSIGNED8 (*Cc_CAN_PUSH) (
  CAN_MSG    *pCAN          // CAN message to transfer
);


/********************************************************************
Main CANcrypt handle with all data required for CANcrypt operation
********************************************************************/
typedef struct
{
  // Identification
  UNSIGNED32  id_1018[4];   // CANopen style identification
                            // Vendor ID, Product Code, Revision, Serial
  // CAN functions
  Cc_CAN_PUSH fct_TxFIFO;   // Push message into transmit FIFO
  Cc_CAN_PUSH fct_TxNOW;    // Transmit message now, bypass transmit FIFO
  // Event call back function
  Cc_CB_EVENT fctcb_event;  // Call back function: system status changed

  // Call back event triggered in interrupt
  UNSIGNED8   event_irq[Cc_CBEVENT_QUEUE]; // Events reported in Process_Rx functions
  UNSIGNED32  event_par[Cc_CBEVENT_QUEUE*2]; // Parameters

  // CAN messages for configurator, device, identify
  CAN_MSG     can_config;   // configurator message
  CAN_MSG     can_device;   // device message
  CAN_MSG     can_ident;    // (Xtd) identifiation message

  // Identify
  UNSIGNED8   identify[4];  // identificiation request received
                            // [0-2] data; [3] 1: regular, 2: extended
  // Public information
  UNSIGNED8   my_address;   // 1 for configurator or 2 to 15 for a device
  UNSIGNED8   my_status;    // CANcrypt device status
  UNSIGNED8   err_counter;  // CANcrypt error counter

  // Internal state machines
#ifdef Cc_USE_PAIRING
  CCkeySTATE_TYPE state_key; // key generation state of this node
  CCparSTATE_TYPE state_pair; // pairing state of this node
#endif
#ifdef Cc_USE_GROUPING
  CCgrpSTATE_TYPE state_group;// grouping state of this node
#endif
#ifdef Cc_USE_MONITORING
  Cc_CANIDLIST_HANDLE unx_lst;// unexpected CAN messages
#endif

  // Key management
  UNSIGNED32    key_dyn[Cc_KEY_LEN32*2]; // current dynamic key and copy of last
  UNSIGNED8     key_rnd[48];  // random values from grouping, sec HB, pairing
  UNSIGNED32   *pkey_perm;    // permanent key used, pointer to it
  CC_TIMER_TYPE timer_delay;  // global delay timer currently in use
  UNSIGNED16    cycle_timeout;// timeout for the cycle time, 
  UNSIGNED8     key_up_cnt;   // count dynamic key updates
  UNSIGNED8     key_method;   // Key generation method, default Cc_METHOD

#ifdef Cc_USE_PAIRING
  UNSIGNED32    key_work;     // current key section to work on
  UNSIGNED32    flipper;      // flip value for key above
  CC_TIMER_TYPE timer_rand;   // random delay timer currently in use
  CC_TIMER_TYPE timer_cycle;  // cycle timer currently in use
  UNSIGNED8     cycle_timernd;// random part of cycle time,
                              // 0x0F: 0-15ms, 0x1F: 0-32ms, 0x3F: 0-64ms
  UNSIGNED8     key_id_mm;    // ID of the key we currently work on, major/minor
  UNSIGNED8     key_size;     // desired size of current key
  UNSIGNED8     key_bit_pos;  // bit position in work_key
  UNSIGNED8     bit_claim;    // next bit we try to claim, 0xFF for not set
  UNSIGNED8     bit_next;     // next bit determined, 0xFF for not set
  UNSIGNED8     bit_ignore;   // counter if bit from a cycle is ignored
#endif

#ifdef Cc_USE_GROUPING
  // Grouping
  CC_TIMER_TYPE grp_time;     // secure heartbeat timer
  UNSIGNED16    grp_HBevent;  // secure heartbeat event time value
  UNSIGNED16    grp_HBinhib;  // secure heartbeat inhibit time value
  UNSIGNED16    grp_hb_seq;
  UNSIGNED16    grp_info_use; // devices in use, found in current cycle
  UNSIGNED16    grp_info_fnd; // ..bit 1 to 15 set if device is in grp
  // keep group below together, this data not erased on re-start
  UNSIGNED8     grp_key_id_mm;// ID of the key we currently use, major/minor
  UNSIGNED8     grp_key_ctrl; // last used major key, highest bit set during key generation
  UNSIGNED16    grp_info_exp; // devices expected, optional 
  UNSIGNED16    grp_info_opt; // ..bit 1 to 15 set if device is in grp
#endif

#ifdef Cc_USE_SECURE_MSG
  // Secure messaging
  Cc_SEC_MSG_TABLE_ENTRY *pMsgTblRx; // secure messages to receive
  Cc_SEC_MSG_TABLE_ENTRY *pMsgTblTx; // secure messages to transmit
  UNSIGNED16             *pMsgCntRx; // list of receive counters
  UNSIGNED16             *pMsgCntTx; // list of transmit counters
#endif

  // Last entry, version
  UNSIGNED16  id_version;   // CANcrypt ID and version
} Cc_HANDLE;


#endif
/*----------------------- END OF FILE -----------------------------*/
